/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import static org.junit.Assert.*;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;

import org.junit.Test;

import nz.org.riskscape.engine.Assert;

public class EnumerationTest {

  Enumeration simple = Enumeration.zeroBased("foo", "bar", "baz");
  Enumeration withGaps = Enumeration.zeroBased(null, "foo", "bar", null, "baz");
  Enumeration oneBased = Enumeration.oneBased("foo", "bar", "baz");
  Enumeration mixedCase = Enumeration.zeroBased("foO", "BAR", "Baz");

  @Test
  public void canCoerceANumberToAString() {
    assertEquals("foo", simple.coerce(0));
    assertEquals("baz", oneBased.coerce(3));
    assertEquals("bar", withGaps.coerce(2));
    assertEquals("Baz", mixedCase.coerce(2));
  }

  @Test
  public void canCoerceANumberAsStringToAString() {
    assertEquals("foo", simple.coerce("0"));
    assertEquals("baz", oneBased.coerce("3"));
    assertEquals("bar", withGaps.coerce("2"));
    assertEquals("Baz", mixedCase.coerce("2"));
    Assert.assertThrows(CoercionException.class, () -> simple.coerce("0.0"));
    Assert.assertThrows(CoercionException.class, () -> simple.coerce("4"));   //There is no index 4
    Assert.assertThrows(CoercionException.class, () -> simple.coerce("3a"));
  }

  @Test
  public void canCoerceAStringToAString() {
    assertEquals("foo", oneBased.coerce("foo"));
    assertEquals("foo", simple.coerce("foo"));
    assertEquals("bar", withGaps.coerce("BAR"));
    assertEquals("BAR", mixedCase.coerce("Bar"));
  }

  @Test
  public void throwsErrorIfNumberOutOfBounds() throws Exception {
    CoercionException ex = Assert.assertThrows(CoercionException.class, () -> simple.coerce(-1));
    assertEquals("-1 is out of range (0 <= x < 3)", ex.getMessage());
    Assert.assertThrows(CoercionException.class, () -> simple.coerce(3));
    ex = Assert.assertThrows(CoercionException.class, () -> oneBased.coerce(0));
    assertEquals("0 is out of range (1 <= x < 4)",  ex.getMessage());
    Assert.assertThrows(CoercionException.class, () -> withGaps.coerce(3));
  }

  @Test
  public void throwsErrorIfStringOutOfSet() throws Exception {
    Assert.assertThrows(CoercionException.class, () -> simple.coerce("foos"));
    Assert.assertThrows(CoercionException.class, () -> withGaps.coerce("bars"));
    Assert.assertThrows(CoercionException.class, () -> withGaps.coerce("BAZz"));
  }

  @Test
  public void canSerializeInAndOut() throws Exception {
    ByteArrayOutputStream bos = new ByteArrayOutputStream();
    DataOutputStream dos = new DataOutputStream(bos);

    simple.toBytes(dos, 1);
    withGaps.toBytes(dos, "foo");
    mixedCase.toBytes(dos, "BAZ");
    oneBased.toBytes(dos, "foo");

    byte[] bytes = bos.toByteArray();

    ByteArrayInputStream bin = new ByteArrayInputStream(bytes);
    DataInputStream din = new DataInputStream(bin);

    assertEquals(1, din.readInt());
    assertEquals(1, din.readInt());
    assertEquals(2, din.readInt());
    assertEquals(1, din.readInt());

    bin.reset();

    assertEquals("bar", simple.fromBytes(din));
    assertEquals("foo", withGaps.fromBytes(din));
    assertEquals("Baz", mixedCase.fromBytes(din));
    assertEquals("foo", oneBased.fromBytes(din));
  }

  @Test(expected=IllegalArgumentException.class)
  public void willFailToConstructOnDuplicateMembers() throws Exception {
    Enumeration.zeroBased("foo", "bar", "foo");
  }

  @Test(expected=IllegalArgumentException.class)
  public void willFailToConstructOnDuplicateMembersPlusNulls() throws Exception {
    Enumeration.zeroBased(null, "foo", null, null, "bar", null, "foo");
  }

  @Test(expected=IllegalArgumentException.class)
  public void willFailToConstructOnDuplicateCaseInsensitiveMembers() throws Exception {
    Enumeration.zeroBased("FOO", "foo");
  }


}
