/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.task;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Collections;

import org.junit.Test;

import nz.org.riskscape.engine.Assert;

public class TaskSpecDependencyTest {

  TaskSpec spec1 = newSpec(ChainTask.class);
  TaskSpec spec2 = newSpec(TupleInputTask.class);
  TaskSpec spec3 = newSpec(SinkTask.class);

  @Test
  public void canConstructATaskSpecAndAssignAndClearItsDependencies() {

    // sanity checks
    assertTrue(spec1.getDependsOn().isEmpty());
    assertTrue(spec2.getDependsOn().isEmpty());
    assertFalse(spec1.hasOutstandingDependencies());
    assertFalse(spec2.hasOutstandingDependencies());

    // add dependency
    spec2.addDependency(spec1);
    spec2.addDependency(spec3);
    assertTrue(spec2.hasOutstandingDependencies());
    assertTrue(spec2.hasDependency(spec1));
    assertTrue(spec2.hasDependency(spec3));
    assertFalse(spec1.hasDependency(spec2));

    // clear one dependency
    spec2.satisfyDependency(spec1);
    assertFalse(spec2.hasDependency(spec1));
    assertTrue(spec2.hasDependency(spec3));
    assertFalse(spec1.hasDependency(spec2));
    assertTrue(spec2.hasOutstandingDependencies());

    // clear the other dependency
    spec2.satisfyDependency(spec3);
    assertFalse(spec2.hasDependency(spec1));
    assertFalse(spec2.hasDependency(spec3));
    assertFalse(spec2.hasOutstandingDependencies());
  }

  @Test
  public void canRegisterAndAccessProcessingResults() throws Exception {
    // nada
    assertThat(spec1.getProcessingResultsFrom(spec2.getWorkerTaskClass(), String.class), empty());

    // add a single result from spec2
    spec1.addProcessingResultFromDependency(spec2, "foo");
    assertThat(spec1.getProcessingResultsFrom(spec2.getWorkerTaskClass(), String.class), contains("foo"));
    assertThat(spec1.getProcessingResultFrom(spec2.getWorkerTaskClass(), String.class), equalTo("foo"));
    assertThat(spec1.getProcessingResultsFrom(spec3.getWorkerTaskClass(), String.class), empty());
    // minimums...
    Assert.assertThrows(IllegalStateException.class, () ->
      spec1.getProcessingResultsFrom(spec2.getWorkerTaskClass(), String.class, 2));
    Assert.assertThrows(IllegalStateException.class, () ->
      spec1.getProcessingResultsFrom(spec3.getWorkerTaskClass(), String.class, 1));

    // add a single result from spec3, check this doesn't affect the results for spec2
    spec1.addProcessingResultFromDependency(spec3, "bar");

    assertThat(spec1.getProcessingResultsFrom(spec2.getWorkerTaskClass(), String.class), contains("foo"));
    assertThat(spec1.getProcessingResultFrom(spec2.getWorkerTaskClass(), String.class), equalTo("foo"));
    assertThat(spec1.getProcessingResultsFrom(spec3.getWorkerTaskClass(), String.class), contains("bar"));

    // add another result from spec2, make sure it is returned and causes getProcessingResultFrom to fail
    spec1.addProcessingResultFromDependency(spec2, "baz");
    assertThat(spec1.getProcessingResultsFrom(spec2.getWorkerTaskClass(), String.class), contains("foo", "baz"));
    Assert.assertThrows(IllegalStateException.class, ()
        -> assertThat(spec1.getProcessingResultFrom(spec2.getWorkerTaskClass(), String.class), equalTo("foo")));
    assertThat(spec1.getProcessingResultsFrom(spec3.getWorkerTaskClass(), String.class), contains("bar"));

  }

  private TaskSpec newSpec(Class<? extends WorkerTask> impl) {
    return new TaskSpec(impl, Collections.emptyList(), null, null, false, null);
  }

}
