/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.task;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.stream.Collectors;

import org.junit.Test;

import nz.org.riskscape.engine.pipeline.RealizedStep;
import nz.org.riskscape.engine.pipeline.sink.ListSink;
import nz.org.riskscape.engine.sched.BaseTaskTest;

public class SinkTaskTest extends BaseTaskTest<SinkTask> {

  private ListSink sink;

  public SinkTaskTest() {
    super(SinkTask.class);
  }

  @Test
  public void readsTuplesFromTheInputUntilExhausted() throws Exception {
    // lots of little pages
    input = newBuffer(2, 100);

    fillInput();
    assertEquals(ReturnState.COMPLETE, createTask(newStep("sink", ListSink.CONSTRUCTOR)).run());
    assertThat(sink.getTuples(), equalTo(longs.iterator().collect(Collectors.toList())));
  }


  @Test
  public void readsTuplesFromTheInputUntilExhaustedWithInterruptions() throws Exception {
    // lots of little pages
    input = newBuffer(2, 100);

    fill(input, longs(0, 5).iterator(), false);
    assertEquals(ReturnState.INPUT_EMPTY, createTask(newStep("sink", ListSink.CONSTRUCTOR)).run());
    fill(input, longs(6, 10).iterator(), true);
    assertEquals(ReturnState.COMPLETE, workerTask.run());

    assertThat(sink.getTuples(), equalTo(longs(0, 10)));
  }

  @Test
  public void readsTuplesUntilTheOutputIsBlocked() throws Exception {
    input = newBuffer(2, 100);
    fill(input, longs(0, 10).iterator(), true);

    createTask(newStep("sink", ListSink.bounded(3)));

    assertEquals(ReturnState.OUTPUT_FULL, workerTask.run());
    assertFalse(sink.canAccept());
    assertEquals(ReturnState.OUTPUT_FULL, workerTask.run());

    sink.capacity = 6;
    assertTrue(sink.canAccept());
    assertEquals(ReturnState.OUTPUT_FULL, workerTask.run());

    sink.capacity = 20;
    assertEquals(ReturnState.COMPLETE, workerTask.run());
    assertThat(sink.getTuples(), equalTo(longs(0, 10)));
  }

  @Override
  public SinkTask createTask(RealizedStep... steps) {
    super.createTask(steps);

    sink = (ListSink) workerTask.sink;

    return workerTask;
  }

}
