/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.steps;

import static nz.org.riskscape.engine.Matchers.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import org.junit.Test;

import com.google.common.collect.Range;

import nz.org.riskscape.engine.problem.GeneralProblems;
import nz.org.riskscape.engine.projection.Projector;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.Problem.Severity;
import nz.org.riskscape.problem.ResultOrProblems;


@SuppressWarnings("unchecked")
public class UnionStepTest extends BaseStepTest<UnionStep> {

  @Override
  protected UnionStep createStep() {
    return new UnionStep(engine);
  }

  @Test
  public void unionStepMustAlwaysHaveInput() throws Exception {
    assertThat(step.realizeSimple(input()), failedResult(
        is(GeneralProblems.get().badArity(Input.class, step, Range.atLeast(2), 0))
    ));
  }

  @Test
  public void getWarningIfUnionStepOnlyHasOneInput() throws Exception {
    // allow this for testing/debugging but we should get a warning
    addStubInput(Struct.of("foo", Types.TEXT));
    assertThat((ResultOrProblems<Projector>) step.realizeSimple(input()), resultWithProblems(
        projecting(Struct.of("foo", Types.TEXT)),
        contains(
            is(GeneralProblems.get().badArity(Input.class, step, Range.atLeast(2), 1).withSeverity(Severity.WARNING))
    )));
  }

  @Test
  public void canCreateASimpleUnionProjector() throws Exception {
    // most of logic and tests for union step are in UnionProjector/Test, but here's a sanity-check
    addStubInput("foo", Struct.of("foo", Types.POLYGON, "bar", Types.TEXT));
    addStubInput("bar", Struct.of("foo", Types.POINT, "baz", Types.INTEGER));
    Struct expected = Struct.of("foo", Types.GEOMETRY, "bar", Nullable.TEXT, "baz", Nullable.INTEGER);
    assertThat(step.realizeSimple(input()), result(Projector.class, projecting(expected)));
  }
}
