/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.steps;

import static nz.org.riskscape.engine.Matchers.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import org.junit.Test;

import nz.org.riskscape.engine.Matchers;
import nz.org.riskscape.engine.bind.ParamProblems;
import nz.org.riskscape.engine.projection.Projector;
import nz.org.riskscape.engine.relation.EmptyRelation;
import nz.org.riskscape.engine.relation.Relation;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.rl.ast.Expression;
import nz.org.riskscape.rl.ast.ExpressionProblems;
import nz.org.riskscape.rl.ast.StructDeclaration;

@SuppressWarnings("unchecked")
public class SelectStepTest extends BaseStepTest<SelectStep> {

  Struct inputType = Struct.of("text", Types.TEXT, "int", Types.INTEGER);
  Relation relation = new EmptyRelation(inputType);

  @Override
  protected SelectStep createStep() {
    return new SelectStep(engine);
  }

  @Test
  public void canRemoveAttributes() throws Exception {
    addInput(relation);
    // NB - we no longer need to pass a struct expression, anything given will be converted to a struct expression
    addParam("select", parse("text"));

    assertThat(
        step.realizeSimple(input()),
        result(Projector.class, projecting(Struct.of("text", Types.TEXT))));
  }

  private Expression parse(String source) {
    return expressionParser.parse(source);
  }

  @Test
  public void canChangeAttributeNames() throws Exception {
    addInput(relation);
    addParam("select", parse("{text as new_text, int as new_int}"));

    assertThat(
        step.realizeSimple(input()),
        result(Projector.class, projecting(Struct.of("new_text", Types.TEXT, "new_int", Types.INTEGER))));
  }

  @Test
  public void errorIfInputMissingMappings() throws Exception {
    addInput(relation);
    Expression expr = parse("{bad_text as new_text, int as new_int}");
    addParam("select", expr);

    assertThat(step.realizeSimple(input()), failedResult(
      Matchers.equalIgnoringChildren(ExpressionProblems.get().failedToRealize(expr, inputType))
    ));
  }

  @Test
  public void errorIfDuplicateSelectMappings() throws Exception {
    addInput(relation);
    StructDeclaration expr = parse("{text as dup, int as dup}").isA(StructDeclaration.class).get();
    addParam("select", expr);

    assertThat(step.realizeSimple(input()), failedResult(
      // no matcher to allow a separate child matcher, but this does a similar thing
      allOf(
        Matchers.equalIgnoringChildren(ExpressionProblems.get().failedToRealize(expr, inputType)),
        Matchers.hasAncestorProblem(
          equalTo(ExpressionProblems.get().canNotReplaceMember("dup", expr.getMember(0), expr.getMember(1)))
        )
      )
    ));
  }

  @Test
  public void arityErrorIfTooManyParametersGiven() throws Exception {
    addInput(relation);

    assertEquals(1, step.getDeclaredParameters().get(0).getMaxRequired());
    assertEquals(1, step.getDeclaredParameters().get(0).getMinRequired());

    addParam("select", parse("{}"));
    addParam("select", parse("{}"));

    assertThat(
      step.realizeSimple(input()),
      failedResult(equalTo(ParamProblems.get().wrongNumberGiven("select", "1", 2)))
    );
  }

}
