/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.steps;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import static nz.org.riskscape.engine.Matchers.*;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import nz.org.riskscape.engine.output.Format;
import nz.org.riskscape.engine.problem.ProblemPlaceholder;
import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.bind.impl.NumberBinder;
import nz.org.riskscape.engine.output.TestFormat;
import nz.org.riskscape.engine.output.TestFormat.TestFormatOptions;
import nz.org.riskscape.engine.pipeline.SinkConstructor;
import nz.org.riskscape.engine.problem.GeneralProblems;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.Expression;

public class SaveStepTest extends BaseStepTest<SaveStep> {

  Struct inputType = Struct.of("geom", Types.TEXT, "value", Types.INTEGER);
  TestFormatOptions expectedOptions = new TestFormatOptions();

  @Override
  protected SaveStep createStep() {
    return new SaveStep(engine);
  }

  @Before
  public void addTestFormat() {
    engine.getFormats().add(new TestFormat(opt -> {}));
  }

  @Test
  public void canGiveFormatSpecificOptions() {
    addParam("format", "test");
    addParam("options", expressionParser.parse("{required: 'foobar'}"));

    expectedOptions.required = "foobar";

    assertThat(realize(), result(
        allOf(
            hasProperty("formatOptions", isPresent(expectedOptions))
        )
    ));
  }

  @Test
  public void spuriousOptionsAreNoticed() {
    addParam("format", "test");
    addParam("options", expressionParser.parse("{required: 'foobar', spurious: 'baz'}"));

    expectedOptions.required = "foobar";
    ProblemPlaceholder options = ProblemPlaceholder.of(Format.FormatOptions.class, "test");

    assertThat(realize(), failedResult(
        is(Problems.foundWith(Format.FormatOptions.class, "test",
            GeneralProblems.get().notAnOption("spurious", options, List.of("required", "number", "expression"))))
    ));
  }

  @Test
  public void errorIfOptionsNotRightType() {
    addParam("format", "test");
    addParam("options", expressionParser.parse("{required: 'foobar', number: 'four'}"));

    assertThat(realize(), failedResult(
        hasAncestorProblem(is(NumberBinder.PROBLEMS.numberFormatException("four", Integer.class)))
    ));
  }

  @Test
  public void optionsCanContainExpressions() {
    addParam("format", "test");
    Expression optionExpr = expressionParser.parse("{required: 'foobar', expression: attr}");
    addParam("options", optionExpr);

    expectedOptions.required = "foobar";
    expectedOptions.expression = Optional.of(expressionParser.parse("attr"));

    assertThat(realize(), result(
        allOf(
            hasProperty("formatOptions", isPresent(expectedOptions))
        )
    ));
  }

  @Test
  public void errorIfOptionsSpecifiedWithoutFormat() {
    addParam("options", expressionParser.parse("{required: 'foobar'}"));

    assertThat(realize(), failedResult(
        is(SaveStep.PROBLEMS.optionsWithNoFormat())
    ));
  }

  @Test
  public void errorWhenOptionsAreRequiredButNotSpecified() {
    // TestFormatOptions.required is required, so we get an error if the options parameter is omitted
    addParam("format", "test");

    assertThat(realize(), failedResult(
        is(SaveStep.PROBLEMS.optionsRequired(engine.getFormats().get("test"), Arrays.asList("required")))
    ));
  }

  @Test
  public void emptyOptionsIsNoOptions() {
    // we treat an empty struct as no options to allow things like
    // `save('foo', format: 'cvs', options: {})` where we'd really expect the format and options to come
    // from pipeline paramters
    addParam("options", expressionParser.parse("{}"));

    assertThat(realize(), result(
        allOf(
            hasProperty("formatOptions", isEmptyOptional())
        )
    ));
  }

  @SuppressWarnings("unchecked")
  private ResultOrProblems<SinkConstructor> realize() {
    addStubInput(inputType);
    return (ResultOrProblems<SinkConstructor>)step.realizeSimple(input());
  }
}
