/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.steps;

import static nz.org.riskscape.engine.Matchers.*;
import static nz.org.riskscape.engine.TupleMatchers.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;

import nz.org.riskscape.engine.RelationMatchers;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.bind.ParamProblems;
import nz.org.riskscape.engine.function.IdentifiedFunction;
import nz.org.riskscape.engine.pipeline.RealizationInput;
import nz.org.riskscape.engine.relation.ListRelation;
import nz.org.riskscape.engine.relation.Relation;
import nz.org.riskscape.engine.types.RelationType;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.types.WithMetadata;
import nz.org.riskscape.problem.Problem.Severity;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ExpressionParser;
import nz.org.riskscape.rl.ast.Expression;
import nz.org.riskscape.rl.ast.ExpressionProblems;

public class RelationInputStepTest extends BaseStepTest<RelationInputStep> {

  @Override
  protected RelationInputStep createStep() {
    return new RelationInputStep(engine);
  }

  ListRelation values = ListRelation.ofValues("foo");

  @Before
  public void setupRelation() {
    addPickledData("foo", values);
  }

  @Test
  public void producesTheRelationFromParameters() {
    addParam("relation", "'foo'");
    assertThat(
         step.realizeSimple(input()),
         result(sameInstance(values)));
  }

  @Test
  public void canOptionallyNestTuples() throws Exception {
    addParam("relation", "'foo'");
    addParam("name", "nest-me");
    Struct expected = Types.TEXT.asStruct().parent("nest-me");

    assertThat(step.realizeSimple(input()),
        result(Relation.class, RelationMatchers.relationWithTuples(Tuple.ofValues(expected,
            Tuple.ofValues(Types.TEXT.asStruct(), "foo")))));
  }

  @Test
  public void canOptionallyApplyALimit() throws Exception {
    addParam("relation", "'foo'");
    addParam("limit", 3L);

    Relation result = (Relation) step.realizeSimple(input()).get();
    assertEquals(3L, result.getLimit());
    assertEquals(0, result.getOffset());
  }

  @Test
  public void canOptionallyApplyAnOffset() throws Exception {
    addParam("relation", "'foo'");
    addParam("offset", 2L);

    Relation result = (Relation) step.realizeSimple(input()).get();
    assertEquals(Long.MAX_VALUE, result.getLimit());
    assertEquals(2L, result.getOffset());

  }

  @Test
  public void canOptionallyApplyALimitAndOffset() throws Exception {
    addParam("relation", "'foo'");
    addParam("offset", 2L);
    addParam("limit", 3L);

    Relation result = (Relation) step.realizeSimple(input()).get();
    assertEquals(2L, result.getOffset());
    assertEquals(3L, result.getLimit());
  }

  @Test
  public void reliesOnBookmarkBinderForCreatingARelation() throws Exception {
    /*
     *  This test mocks up the binding context to assert it uses that for creating a relation - this is important
     *  because we know we've covered off testing how that behaves elsewhere (so we don't need to re-test it)
     */
    bindingContext = Mockito.mock(BindingContext.class);
    // use 111 here because it's a valid constant expression, but it isn't a text type
    // and isn't an existing bookmark ID. This should get passed to the binder as is
    addParam("relation", "111");

    when(bindingContext.bind("111", Relation.class)).thenReturn(ResultOrProblems.of(values));

    RealizationInput mockInput = mock(RealizationInput.class);
    when(mockInput.getBindingContext()).thenReturn(bindingContext);
    when(mockInput.getRealizationContext()).thenReturn(realizationContext);

    RelationInputStep.Parameters params = new RelationInputStep.Parameters();
    params.relation = Optional.of(ExpressionParser.INSTANCE.parse("111"));
    params.value = Optional.empty();
    params.input = mockInput;
    params.name = Optional.empty();

    Relation result = (Relation) step.realize(params).get();
    assertSame(result, values);

    verify(bindingContext).bind("111", Relation.class);

    when(bindingContext.bind("111", Relation.class))
        .thenReturn(ResultOrProblems.failed(Problems.foundWith("failure")));
    assertThat(
      step.realize(params),
      failedResult(equalTo(Problems.foundWith("relation", Problems.foundWith("failure"))))
    );
  }

  @Test
  public void canBeAnExpressionThatYieldsABookmark() throws Exception {
    // need a pickled version of the LookupBookmark function  (it's in a different package)
    project.getFunctionSet().add(new IdentifiedFunction() {

      @Override
      public List<Type> getArgumentTypes() {
        return Arrays.asList(Types.TEXT);
      }

      @Override
      public Type getReturnType() {
        return WithMetadata.wrap(RelationType.WILD, project.getBookmarks().get("foo"));
      }

      @Override
      public Object call(List<Object> args) {
        return values;
      }

      @Override
      public String getId() {
        return "bookmark";
      }

      @Override
      public String getDescription() {
        return null;
      }

      @Override
      public String getSource() {
        return null;
      }

      @Override
      public Category getCategory() {
        return Category.LANGUAGE;
      }
    });

    addParam("relation", "bookmark('foo')");
    assertThat(step.realizeSimple(input()), result(sameInstance(values)));
  }

  @Test
  public void canOptionallySupplyANestedValueInsteadOfARelation() throws Exception {
    addParam("value", "'foo'");
    addParam("name", "bar");

    assertThat(
      step.realizeSimple(input()),
      result(Relation.class, RelationMatchers.relationWithTuples(
        contains(tupleWithValue("bar",  equalTo("foo")))
      ))
    );
  }

  @Test
  public void valueCanBeASingleTuple() throws Exception {
    addParam("value", "{foo: 'bar', baz: 1}");

    assertThat(
      step.realizeSimple(input()),
      result(Relation.class, RelationMatchers.relationWithTuples(contains(
        allOf(
          tupleWithValue("foo", equalTo("bar")),
          tupleWithValue("baz", equalTo(1L))
        )
      )))
    );
  }

  @Test
  public void valueCanBeASingleNestedTuple() throws Exception {
    addParam("value", "{foo: 'bar', baz: 1}");
    addParam("name", "nested");

    assertThat(
      step.realizeSimple(input()),
      result(Relation.class, RelationMatchers.relationWithTuples(contains(
        tupleWithValue("nested",
          allOf(
            tupleWithValue("foo", equalTo("bar")),
            tupleWithValue("baz", equalTo(1L))
          )
        )
      )))
    );
  }

  @Test
  public void eitherValueOrRelationMustBeGiven() throws Exception {
    assertThat(
      step.realizeSimple(input()),
      failedResult(equalTo(ParamProblems.oneOfTheseRequired("relation", "value")))
    );

    addParam("value", "1");
    addParam("relation", "'foo'");

    assertThat(
      step.realizeSimple(input()),
      failedResult(equalTo(ParamProblems.get().mutuallyExclusive("relation", "value")))
    );

  }

  @Test
  public void givesAnErrorIfValueDoesNotRealize() throws Exception {
    Expression nonConstant = ExpressionParser.INSTANCE.parse("{foo: bar}");
    addParam("value", nonConstant.toSource());

    assertThat(
      step.realizeSimple(input()),
      failedResult(equalTo(
        Problems.foundWith("value", ExpressionProblems.get().constantRequired(nonConstant))))
    );
  }

  @Test
  public void givesWarningsIfLimitAndOffsetUsedWithAValue() throws Exception {
    addParam("value", ExpressionParser.INSTANCE.parse("'foo'"));
    addParam("limit", 1L);
    addParam("offset", 1L);

    ResultOrProblems<Relation> realizedOr = (ResultOrProblems<Relation>) step.realizeSimple(input());
    assertThat(realizedOr, resultWithProblems(
        RelationMatchers.relationWithTuples(
          contains(tupleWithValue("value", equalTo("foo")))
        ),
        containsInAnyOrder(
          ParamProblems.get().ignored("limit").withSeverity(Severity.WARNING),
          ParamProblems.get().ignored("offset").withSeverity(Severity.WARNING)
       ))
    );
  }

}
