/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.steps;

import static nz.org.riskscape.engine.Matchers.*;
import static org.junit.Assert.*;

import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeDiagnosingMatcher;
import org.junit.Test;

import nz.org.riskscape.engine.join.LopsidedJoiner;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.join.JoinType;
import nz.org.riskscape.engine.join.ProblemCodes;
import nz.org.riskscape.engine.relation.ListRelation;
import nz.org.riskscape.engine.relation.Relation;
import nz.org.riskscape.engine.relation.TupleIterator;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.problem.Problem.Severity;
import nz.org.riskscape.rl.ast.Expression;

@SuppressWarnings("unchecked")
public class JoinStepTest extends BaseStepTest<JoinStep> {

  @Override
  protected JoinStep createStep() {
    return new JoinStep(engine);
  }

  @Test
  public void canJoinTwoInputs() throws Exception {
    Struct lhsType = Struct.of("foo", Types.TEXT);
    Relation lhs = new ListRelation(lhsType, Tuple.ofValues(lhsType, "cat"), Tuple.ofValues(lhsType, "dog"));

    Struct rhsType = Struct.of("bar", Types.TEXT);
    Relation rhs = new ListRelation(rhsType, Tuple.ofValues(rhsType, "cat"), Tuple.ofValues(rhsType, "bird"));

    Expression joinOn = expressionParser.parse("foo = bar");

    addParam("on", joinOn);

    addInput("lhs", lhs);
    addInput("rhs", rhs);

    Struct resultType = Struct.of("foo", Types.TEXT, "bar", Types.TEXT);

    ResultOrProblems<LopsidedJoiner> joiner = (ResultOrProblems<LopsidedJoiner>) step.realizeSimple(input());
    assertThat(joiner, result(LopsidedJoiner.class, joinerProducingType(resultType)));

    Object index = primeIndex(joiner.get(), rhs);
    assertThat(joiner.get().join(index, Tuple.ofValues(lhsType, "cat")), iteratorWithTuples(
        Tuple.ofValues(resultType, "cat", "cat")
    ));

    assertThat(joiner.get().join(index, Tuple.ofValues(lhsType, "dog")), iteratorWithTuples(
    ));
  }

  @Test
  public void canJoinTwoInputs_LeftOuter() throws Exception {
    Struct lhsType = Struct.of("foo", Types.TEXT);
    Relation lhs = new ListRelation(lhsType, Tuple.ofValues(lhsType, "cat"), Tuple.ofValues(lhsType, "dog"));

    Struct rhsType = Struct.of("bar", Types.TEXT);
    Relation rhs = new ListRelation(rhsType, Tuple.ofValues(rhsType, "cat"), Tuple.ofValues(rhsType, "bird"));

    Expression joinOn = expressionParser.parse("foo = bar");

    addParam("on", joinOn);
    addParam("join-type", JoinType.LEFT_OUTER);

    addInput("lhs", lhs);
    addInput("rhs", rhs);

    Struct resultType = Struct.of("foo", Types.TEXT, "bar", Nullable.TEXT);

    ResultOrProblems<LopsidedJoiner> joiner = (ResultOrProblems<LopsidedJoiner>) step.realizeSimple(input());
    assertThat(joiner, result(LopsidedJoiner.class, joinerProducingType(resultType)));

    Object index = primeIndex(joiner.get(), rhs);

    assertThat(joiner.get().join(index, Tuple.ofValues(lhsType, "cat")), iteratorWithTuples(
        Tuple.ofValues(resultType, "cat", "cat")
    ));

    assertThat(joiner.get().join(index, Tuple.ofValues(lhsType, "dog")), iteratorWithTuples(
        Tuple.ofValues(resultType, "dog")
    ));
  }

  @Test
  public void cannotJoinIfLhsAndRhsShareAttributes() {
    Struct lhsType = Struct.of("foo", Types.TEXT);
    Relation lhs = new ListRelation(lhsType, Tuple.ofValues(lhsType, "cat"), Tuple.ofValues(lhsType, "dog"));

    Relation rhs = new ListRelation(lhsType, Tuple.ofValues(lhsType, "cat"), Tuple.ofValues(lhsType, "bird"));

    Expression joinOn = expressionParser.parse("true");

    addParam("on", joinOn);
    addParam("join-type", JoinType.LEFT_OUTER);

    addInput("lhs", lhs);
    addInput("rhs", rhs);

    assertThat(step.realizeSimple(input()), failedResult(
        isProblem(Severity.ERROR, ProblemCodes.DUPLICATE_KEYS)
    ));
  }

  private Object primeIndex(LopsidedJoiner joiner, Relation rhs) {
    Object index = joiner.createRhsIndex();
    TupleIterator rightSide = rhs.iterator();
    while (rightSide.hasNext()) {
      joiner.addToRhsIndex(rightSide.next(), index);
    }
    return index;
  }

  private Matcher<LopsidedJoiner> joinerProducingType(Struct expectedType) {
    return new TypeSafeDiagnosingMatcher<LopsidedJoiner>(LopsidedJoiner.class) {

      @Override
      public void describeTo(Description description) {
        description.appendText("Joiner producing ").appendValue(expectedType);
      }

      @Override
      protected boolean matchesSafely(LopsidedJoiner item, Description mismatchDescription) {
        if (!item.getJoinedType().equals(expectedType)) {
          mismatchDescription.appendText("producing ").appendValue(item.getJoinedType());
          return false;
        } else {
          return true;
        }
      }
    };
  }

}
