/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.steps;

import java.util.List;
import java.util.Map;

import org.junit.Before;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.bind.Parameter;
import nz.org.riskscape.engine.function.StringFunctions;
import nz.org.riskscape.engine.pipeline.ExecutionContext;
import nz.org.riskscape.engine.pipeline.RealizationInputImpl;
import nz.org.riskscape.engine.pipeline.RealizedPipeline;
import nz.org.riskscape.engine.pipeline.RealizedStep;
import nz.org.riskscape.engine.pipeline.TestExecutionContext;
import nz.org.riskscape.engine.relation.EmptyRelation;
import nz.org.riskscape.engine.relation.Relation;
import nz.org.riskscape.engine.rl.DefaultOperators;
import nz.org.riskscape.engine.rl.ExpressionRealizer;
import nz.org.riskscape.engine.rl.GeometryFunctions;
import nz.org.riskscape.engine.rl.LogicFunctions;
import nz.org.riskscape.engine.rl.MathsFunctions;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.pipeline.ast.PipelineDeclaration;
import nz.org.riskscape.pipeline.ast.StepDefinition;
import nz.org.riskscape.rl.ExpressionParser;
import nz.org.riskscape.rl.ast.Expression;

public abstract class BaseStepTest<T extends BaseStep<?>> extends ProjectTest {

  public ExpressionParser expressionParser = new ExpressionParser();

  public Map<String, List<?>> parameters = Maps.newHashMap();
  // used to track whether addParam should clear out a a default from the parameters field
  public Map<String, List<?>> defaultParameters = Maps.newHashMap();
  public List<RealizedStep> inputs = Lists.newArrayList();
  public T step;
  public ExecutionContext context;

  public RealizedPipeline realizedPipeline;

  @Before
  public final void setupStep() {
    this.project.getFunctionSet().insertFirst(DefaultOperators.INSTANCE);
    GeometryFunctions geometryFunctions = new GeometryFunctions(engine);
    this.project.getFunctionSet().addAll(geometryFunctions.getPredicates());
    this.project.getFunctionSet().addAll(geometryFunctions.getFunctions());
    this.project.getFunctionSet().addAll(LogicFunctions.LOGIC_FUNCTIONS);
    this.project.getFunctionSet().addAll(StringFunctions.FUNCTIONS);
    this.project.getFunctionSet().addAll(MathsFunctions.FUNCTIONS);

    this.context = TestExecutionContext.createDefault(project);

    this.step = createStep();

    for (Parameter param : this.step.getDeclaredParameters()) {
      List<?> defaultValues = param.getDefaultValues(context.getBindingContext());
      if (!defaultValues.isEmpty()) {
        parameters.put(param.getName(), defaultValues);
        defaultParameters.put(param.getName(), defaultValues);
      }
    }

    realizedPipeline = RealizedPipeline.empty(context, PipelineDeclaration.EMPTY);
  }

  protected abstract T createStep();

  @SuppressWarnings("unchecked")
  public void addParam(String key, Object paramValue) {
    if (defaultParameters.containsKey(key)) {
      parameters.remove(key);
      defaultParameters.remove(key);
    }
    parameters.compute(key, (k, v) -> {
      if (v == null) {
        return Lists.newArrayList(paramValue);
      } else {
        List list = v;
        list.add(paramValue);
        return v;
      }
    });
  }

  /**
   * Updates the pipeline with a new input step named 'input' that produces values from the given relation
   */
  protected void addInput(Relation relation) {
    addInput("input", relation);
  }

  /**
   * Updates the pipeline with a new input step that produces values from the given relation
   */
  protected void addInput(String stepName, Relation relation) {
    RealizedStep realizedStep = RealizedStep
        .named(stepName)
        .withResult(relation);

    inputs.add(realizedStep);
    realizedPipeline = realizedPipeline.add(realizedStep);
  }

  /**
   * Updates the pipeline with a new input step that produces the given type
   */
  protected void addStubInput(String stepName, Struct type) {
    addInput(stepName, new EmptyRelation(type));
  }

  /**
   * Updates the pipeline with a new step called 'input' that produces the given type
   */
  protected void addStubInput(Struct type) {
    addStubInput("input", type);
  }

  protected RealizationInputImpl input() {
    return input("unnamed");
  }

  protected RealizationInputImpl input(String stepName) {
    return new RealizationInputImpl(realizedPipeline, new StepDefinition(stepName), stepName, inputs, parameters);
  }

  protected ExpressionRealizer getExpressionRealizer() {
    return context.getExpressionRealizer();
  }

  protected Expression toExpression(String expr) {
    return expressionParser.parse(expr);
  }

}
