/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.sort;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Collections;

import org.junit.Test;

import nz.org.riskscape.engine.sort.SortBy.Direction;
import nz.org.riskscape.rl.ast.Expression;
import nz.org.riskscape.rl.ast.PropertyAccess;

public class SortByTest {

  Expression expr1 = PropertyAccess.of("foo");
  Expression expr2 = PropertyAccess.of("bar");
  Expression expr3 = PropertyAccess.of("baz");

  @Test
  public void testEquality() throws Exception {
    assertEquals(SortBy.NONE, SortBy.NONE);
    assertEquals(new SortBy(expr1, Direction.ASC), new SortBy(expr1, Direction.ASC));
    assertNotEquals(new SortBy(expr1, Direction.ASC), new SortBy(expr1, Direction.DESC));
    assertNotEquals(new SortBy(expr1, Direction.ASC), new SortBy(expr2, Direction.ASC));

    assertEquals(
        new SortBy(expr1, Direction.ASC).andThen(new SortBy(expr2, Direction.ASC)),
        new SortBy(expr1, Direction.ASC).andThen(new SortBy(expr2, Direction.ASC)));

    assertNotEquals(
        new SortBy(expr1, Direction.ASC).andThen(new SortBy(expr2, Direction.ASC)),
        new SortBy(expr1, Direction.ASC).andThen(new SortBy(expr3, Direction.ASC)));
  }

  @Test
  public void andThenAppendsToTheEndNotToTheStart() throws Exception {
    SortBy sortBy = new SortBy(expr1, Direction.ASC).andThen(new SortBy(expr2, Direction.DESC))
        .andThen(new SortBy(expr3, Direction.ASC));

    assertEquals(expr1, sortBy.getExpr());
    assertEquals(expr2, sortBy.getNext().getExpr());
    assertEquals(expr3, sortBy.getNext().getNext().getExpr());
  }

  @Test
  public void canAssembleAListOfSortBysInToAContiguousSortBy() throws Exception {
    SortBy sortBy1 = new SortBy(expr1, Direction.ASC);
    SortBy sortBy2 = new SortBy(expr2, Direction.DESC);
    SortBy sortBy3 = new SortBy(expr3, Direction.DESC);

    SortBy sortBy = SortBy.concatenate(sortBy1, sortBy2, sortBy3);
    assertEquals(expr1, sortBy.getExpr());
    assertEquals(expr2, sortBy.getNext().getExpr());
    assertEquals(expr3, sortBy.getNext().getNext().getExpr());
  }

  @Test
  public void canConcatenateListOfSortBys1() throws Exception {
    SortBy sortBy = SortBy.concatenate(new SortBy(expr1, Direction.ASC),
        new SortBy(expr2, Direction.DESC),
        new SortBy(expr3, Direction.ASC)
    );

    assertEquals(expr1, sortBy.getExpr());
    assertEquals(expr2, sortBy.getNext().getExpr());
    assertEquals(expr3, sortBy.getNext().getNext().getExpr());
  }

  @Test
  public void canConcatenateListOfSortBys_FirstSortByHasNext() throws Exception {
    SortBy sortBy = SortBy.concatenate(new SortBy(expr1, Direction.ASC).andThen(new SortBy(expr2, Direction.DESC)),
        new SortBy(expr3, Direction.ASC)
    );

    assertEquals(expr1, sortBy.getExpr());
    assertEquals(expr2, sortBy.getNext().getExpr());
    assertEquals(expr3, sortBy.getNext().getNext().getExpr());
  }

  @Test
  public void canConcatenateListOfSortBys_LastSortByHasNext() throws Exception {
    SortBy sortBy = SortBy.concatenate(new SortBy(expr1, Direction.ASC),
        new SortBy(expr2, Direction.DESC).andThen(new SortBy(expr3, Direction.ASC))
    );

    assertEquals(expr1, sortBy.getExpr());
    assertEquals(expr2, sortBy.getNext().getExpr());
    assertEquals(expr3, sortBy.getNext().getNext().getExpr());
  }

  @Test
  public void concatenateWithEmptyListReturnsNone() {
    assertThat(SortBy.concatenate(), is(SortBy.NONE));
    assertThat(SortBy.concatenate(Collections.emptyList()), is(SortBy.NONE));
  }

}
