/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.sched;

import java.util.ArrayList;
import java.util.List;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.task.ReturnState;
import nz.org.riskscape.engine.task.TaskSpec;
import nz.org.riskscape.engine.task.WorkerTask;

// simple worker that just reads tuples and stores them
public class TestTask extends WorkerTask {
  List<Tuple> tuplesRead = new ArrayList<Tuple>();
  ReturnState returnState;

  // so we can get a task to block on demand
  private Object sleepMutex = new Object();
  private long tupleReadBlockThreshold = Long.MAX_VALUE;

  public TestTask(TaskSpec spec) {
    super(spec);
  }

  @Override
  public String toString() {
    return super.toString() + " returnState=" + returnState;
  }

  private void waitForUnblock() {
    synchronized (sleepMutex) {
      try {
        sleepMutex.wait();
      } catch (InterruptedException e) {
        throw new RuntimeException("Test interrupted", e);
      }
    }
  }

  private ReturnState runLoop() {
    while (!pageReader.isComplete()) {
      Tuple tuple = pageReader.peek();
      if (tuple == null) {
        // sanity-check just in case the last page got read by another thread between the
        // top of the loop and the peek()
        return pageReader.isComplete() ? ReturnState.COMPLETE : ReturnState.INPUT_EMPTY;
      }

      // so we can artificially slow down/block a task when we
      // want to run several tasks in parallel
      synchronized (sleepMutex) {
        if (tuplesRead.size() >= tupleReadBlockThreshold) {
          waitForUnblock();
        }
      }

      if (pageWriter.isFull()) {
        return ReturnState.OUTPUT_FULL;
      }
      tuplesRead.add(tuple);
      pageWriter.add(tuple);
      pageReader.next();
    }
    return taskComplete();
  }

  @Override
  public ReturnState run() {
    returnState = runLoop();
    return returnState;
  }

  @Override
  public boolean producesResult() {
    return false;
  }

  public void unblock() {
    synchronized (sleepMutex) {
      tupleReadBlockThreshold = Long.MAX_VALUE;
      sleepMutex.notify();
    }
  }

  public void blockAfter(long tupleCount) {
    tupleReadBlockThreshold = tupleCount;
  }
}
