/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.sched;


import static org.junit.Assert.*;

import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.junit.Test;

import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.pipeline.ExecutionContext;
import nz.org.riskscape.engine.pipeline.ExecutionResult;
import nz.org.riskscape.engine.pipeline.RealizedPipeline;
import nz.org.riskscape.engine.pipeline.TestPipelineJobContext;
import nz.org.riskscape.engine.rl.DefaultOperators;
import nz.org.riskscape.engine.rl.LanguageFunctions;
import nz.org.riskscape.engine.rl.MathsFunctions;
import nz.org.riskscape.engine.steps.GroupByStep;
import nz.org.riskscape.engine.steps.JoinStep;
import nz.org.riskscape.engine.steps.RelationInputStep;
import nz.org.riskscape.engine.steps.SelectStep;
import nz.org.riskscape.engine.steps.UnnestStep;
import nz.org.riskscape.pipeline.PipelineParser;
import nz.org.riskscape.pipeline.ast.PipelineDeclaration;
import nz.org.riskscape.problem.Problems;

public class SchedulerBasedExecutorTest extends ProjectTest {

  @Test
  public void canThrowABunchOfJobsAtItAndHaveTheExecutorWorkThroughThemOneByOne() throws Exception {

    engine.getPipelineSteps().add(new SelectStep(engine));
    engine.getPipelineSteps().add(new RelationInputStep(engine));
    engine.getPipelineSteps().add(new JoinStep(engine));
    engine.getPipelineSteps().add(new GroupByStep(engine));
    engine.getPipelineSteps().add(new UnnestStep(engine));

    project.getFunctionSet().insertFirst(new DefaultOperators());
    project.getFunctionSet().addAll(LanguageFunctions.FUNCTIONS);
    project.getFunctionSet().addAll(MathsFunctions.FUNCTIONS);

    final int numExecutions = 1000;
    final int numThreads = 5;

    AtomicInteger counter = new AtomicInteger(0);
    AtomicInteger failureCount = new AtomicInteger(0);

    PipelineDeclaration good = parse(""
        + "input(value: range(0, 50), name: 'lhs') "
        + "-> "
        + "unnest(lhs)"
        + "->"
        + "join.lhs "
        + ""
        + "input(value: range(0, 50), name: 'rhs') "
        + "->"
        + "unnest(rhs)"
        + "-> "
        + "join(on: true).rhs "
        + "-> "
        + "select({lhs + rhs as value})"
        + "->"
        + "group(sum(value))");

    List<PipelineDeclaration> badOnes = Arrays.asList(""
        + "input(value: 'sixty six') "
        + "-> "
        + "select(int(value))",
        "input(value: 'boom') as kabmlamo",
        "" // empty string - not bad per se, but there was a bug with this where it would stall execution
    ).stream()
      .map(this::parse)
      .collect(Collectors.toList());

    List<Future<ExecutionResult>> futures = Collections.synchronizedList(new LinkedList<>());

    // spawn some threads to concurrently adds jobs and then wait for the results to come in
    List<Thread> threads = IntStream.range(0, numThreads).mapToObj(threadCount -> new Thread(() -> {
      int executionsPerThread = numExecutions / numThreads;

      for (int i = 0; i < executionsPerThread; i++) {
        ExecutionContext ec = engine.getPipelineExecutor().newExecutionContext(project);
        RealizedPipeline realized = ec.realize(good);
        futures.add(ec.getPipelineExecutor().execute(new TestPipelineJobContext(realized)));
      }

    }, Integer.toString(threadCount))).collect(Collectors.toList());

    // this thread 'poisons' the queue with bad jobs
    threads.add(new Thread(() -> {
      for (int i = 0; i < 100; i++) {
        ExecutionContext ec = engine.getPipelineExecutor().newExecutionContext(project);
        RealizedPipeline realized = ec.realize(badOnes.get(i % badOnes.size()));
        TestPipelineJobContext jobContext = new TestPipelineJobContext(realized);
        // the one with the kablamp step needs to go kablamo
        if (realized.getStep("kabmlamo").isPresent()) {
          jobContext.getOutputContainer().grenades.put("kabmlamo", Problems.foundWith("kablamo"));
        }

        futures.add(ec.getPipelineExecutor().execute(jobContext));
      }
    }));

    // start 'em off...
    threads.forEach(Thread::start);
    threads.forEach(t -> {
      try {
        t.join();
      } catch (InterruptedException e) {
        e.printStackTrace();
      }
    });

    // count 'em back in
    while (!futures.isEmpty()) {
      Future<ExecutionResult> future = futures.remove(0);

      try {
        future.get();
        counter.incrementAndGet();
      } catch (InterruptedException | ExecutionException e) {
        failureCount.incrementAndGet();
      }
    }

    // add on 33 for the empty pipelines - they aren't actually failures (yet - maybe they should be?)
    assertEquals(numExecutions + 33, counter.get());
    assertEquals(67, failureCount.get());
  }

  PipelineDeclaration parse(String pipelineSource) {
    return PipelineParser.INSTANCE.parsePipeline(pipelineSource);
  }
}
