/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.sched;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.NoSuchElementException;

import org.junit.Test;

import nz.org.riskscape.engine.Assert;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.sched.Page.ReadOnlyPage;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;

public class PageTest {

  private Tuple tuple(String value) {
    return Tuple.of(Struct.of("value", Types.TEXT), "value", value);
  }

  @Test
  public void canWriteTuplesToPage() {
    Page page = new Page(3);
    assertFalse(page.isFull());
    assertEquals(0, page.getTupleCount());
    page.add(tuple("foo"));
    assertFalse(page.isFull());
    assertEquals(1, page.getTupleCount());
    page.add(tuple("bar"));
    assertFalse(page.isFull());
    assertEquals(2, page.getTupleCount());
    page.add(tuple("baz"));
    assertTrue(page.isFull());
    assertEquals(3, page.getTupleCount());

  }

  @Test
  public void canReadTuplesFromPage() {
    Page writeTo = new Page(3);
    writeTo.add(tuple("foo"));
    writeTo.add(tuple("bar"));
    writeTo.add(tuple("baz"));

    ReadOnlyPage readFrom = writeTo.getReadOnlyCopy();
    assertEquals(3, readFrom.getTupleCount());
    assertTrue(readFrom.hasNext());
    assertThat(readFrom.peek(), equalTo(tuple("foo")));
    assertThat(readFrom.next(), equalTo(tuple("foo")));
    assertTrue(readFrom.hasNext());
    assertThat(readFrom.peek(), equalTo(tuple("bar")));
    assertThat(readFrom.next(), equalTo(tuple("bar")));
    assertTrue(readFrom.hasNext());
    assertThat(readFrom.peek(), equalTo(tuple("baz")));
    assertThat(readFrom.next(), equalTo(tuple("baz")));
    assertFalse(readFrom.hasNext());
  }

  @Test
  public void canSharePageBetweenReadThreads() {
    Page writeTo = new Page(3);
    writeTo.add(tuple("foo"));
    writeTo.add(tuple("bar"));
    writeTo.add(tuple("baz"));

    ReadOnlyPage readOne = writeTo.getReadOnlyCopy();
    ReadOnlyPage readTwo = writeTo.getReadOnlyCopy();

    // check we can read from one page and not touch the other
    assertThat(readOne.peek(), equalTo(tuple("foo")));
    assertThat(readOne.next(), equalTo(tuple("foo")));
    assertThat(readTwo.peek(), equalTo(tuple("foo")));
    assertThat(readOne.peek(), equalTo(tuple("bar")));
    assertThat(readOne.next(), equalTo(tuple("bar")));
    assertThat(readTwo.peek(), equalTo(tuple("foo")));

    // read from two, just to mix things up
    assertThat(readTwo.next(), equalTo(tuple("foo")));
    assertThat(readOne.peek(), equalTo(tuple("baz")));

    // finish reading page one
    assertTrue(readOne.hasNext());
    assertTrue(readTwo.hasNext());
    assertThat(readOne.next(), equalTo(tuple("baz")));
    assertThat(readTwo.peek(), equalTo(tuple("bar")));
    assertFalse(readOne.hasNext());
    assertTrue(readTwo.hasNext());

    // finish reading page two
    assertThat(readTwo.next(), equalTo(tuple("bar")));
    assertThat(readTwo.next(), equalTo(tuple("baz")));
    assertFalse(readTwo.hasNext());

    // check we can start again from scratch with a new copy of the page
    ReadOnlyPage readThree = writeTo.getReadOnlyCopy();
    assertEquals(3, readThree.getTupleCount());
    assertTrue(readThree.hasNext());
    assertThat(readThree.peek(), equalTo(tuple("foo")));
  }

  @Test
  public void errorIfOverfillPage() {
    Page page = new Page(1);
    page.add(tuple("foo"));
    Assert.assertThrows(IllegalStateException.class, () -> page.add(tuple("bar")));
  }

  @Test
  public void errorIfOverrunPageRead() {
    Page page = new Page(1);
    page.add(tuple("foo"));
    ReadOnlyPage readFrom = page.getReadOnlyCopy();
    readFrom.next();
    Assert.assertThrows(NoSuchElementException.class, () -> readFrom.peek());
    Assert.assertThrows(NoSuchElementException.class, () -> readFrom.next());
  }
}
