/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.sched;

import static org.junit.Assert.*;

import org.junit.Test;

import lombok.AllArgsConstructor;
import nz.org.riskscape.engine.Assert;

public class PageAllocatorTest {

  @Test
  public void canAllocateFixedPage() {
    PageAllocator allocator = new PageAllocator(5);
    assertEquals(5, allocator.newPage().getMaxSize());
    assertEquals(5, allocator.newPage().getMaxSize());
    assertEquals(5, allocator.newPage().getMaxSize());
  }

  @Test
  public void canGrowPageSize() {
    PageAllocator allocator = new PageAllocator(5).doubleEvery(2);
    assertEquals(5, allocator.newPage().getMaxSize());
    assertEquals(5, allocator.newPage().getMaxSize());
    assertEquals(10, allocator.newPage().getMaxSize());
    assertEquals(10, allocator.newPage().getMaxSize());
    assertEquals(20, allocator.newPage().getMaxSize());
    assertEquals(20, allocator.newPage().getMaxSize());
  }

  @Test
  public void canGrowUpToMaxSize() {
    PageAllocator allocator = new PageAllocator(2).doubleEvery(1).withMaxPageSize(16);
    assertEquals(2, allocator.newPage().getMaxSize());
    assertEquals(4, allocator.newPage().getMaxSize());
    assertEquals(8, allocator.newPage().getMaxSize());
    assertEquals(16, allocator.newPage().getMaxSize());
    assertEquals(16, allocator.newPage().getMaxSize());
  }

  @AllArgsConstructor
  private class BackoffChecker {
    public boolean backoffNeeded;
  }

  @Test
  public void canBackOffPageSize() {
    // check we can reduce the page-size on demand, i.e. backing off from memory pressure
    BackoffChecker backoff = new BackoffChecker(false);
    PageAllocator allocator = new PageAllocator(5)
        .doubleEvery(1)
        .withBackoff(x -> backoff.backoffNeeded);
    assertEquals(5, allocator.newPage().getMaxSize());
    assertEquals(10, allocator.newPage().getMaxSize());
    assertEquals(20, allocator.newPage().getMaxSize());

    backoff.backoffNeeded = true;
    assertEquals(5, allocator.newPage().getMaxSize());
    assertEquals(5, allocator.newPage().getMaxSize());

    backoff.backoffNeeded = false;
    // immediately starts doubling again
    assertEquals(10, allocator.newPage().getMaxSize());
    assertEquals(20, allocator.newPage().getMaxSize());
  }

  @Test
  public void firstPageSizeIsMinOfInitialAndMaxPageSizes() {
    // handle the initial page size being way off in terms of the size we want
    PageAllocator allocator = new PageAllocator(1000).withMaxPageSize(5);
    assertEquals(5, allocator.newPage().getMaxSize());
    assertEquals(5, allocator.newPage().getMaxSize());
    assertEquals(5, allocator.newPage().getMaxSize());
  }

  @Test
  public void canOnlyUseBackoffWithDoublingPageSize() {
    // currently backing off is only supported when the page grows/doubles in size
    BackoffChecker backoff = new BackoffChecker(false);
    PageAllocator allocator = new PageAllocator(5);

    Assert.assertThrows(IllegalStateException.class, () ->  allocator.withBackoff(x -> backoff.backoffNeeded));
  }

}
