/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.sched;

import static nz.org.riskscape.engine.Assert.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.lang.Thread.State;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;

import org.junit.Test;
import org.mockito.Mockito;

import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.engine.output.PipelineJobContext;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;

public class ExecutionFutureTest {

  PipelineJobContext mockJobContext = Mockito.mock(PipelineJobContext.class);
  ExecutionFuture future = new ExecutionFuture(mockJobContext);
  AtomicReference<Object> returned = new AtomicReference<>();
  Object failWith;
  Runnable getCode;

  @Test(timeout = 1000)
  public void canCreateAFutureAndWaitForItToBeFinished() throws Exception {

    failWith = null;

    getCode = () -> {
      try {
        returned.set(future.get());
      } catch (InterruptedException | ExecutionException e) {
        returned.set(e);
        e.printStackTrace();
      }
    };

    waitAndSet();

    assertThat(returned.get(), hasProperty("jobContext", sameInstance(mockJobContext)));
  }

  @Test(timeout = 1000)
  public void canCreateAFutureAndWaitForItToFailDueToAProblem() throws Exception {
    failWith = Problems.foundWith("threading");

    getCode = () -> {
      try {
        future.get();
        returned.set("should have thrown an exception");
      } catch (InterruptedException | ExecutionException e) {
        returned.set(e.getCause());
      }
    };

    waitAndSet();

    assertThat(returned.get(), allOf(
      instanceOf(RiskscapeException.class),
      hasProperty("problem", equalTo(failWith))
    ));
  }

  @Test(timeout = 1000)
  public void timeoutExceptionIfNoResult() {
    // we expect a timeout exception after 100ms if there is no result
    assertThrows(TimeoutException.class, () -> future.get(100, TimeUnit.MILLISECONDS));
  }

  @Test(timeout = 1000)
  public void canCreateAFutureAndWaitForItToFailDueToAnException() throws Exception {

    failWith = new Throwable("aarrgh");

    getCode = () -> {
      try {
        future.get();
        returned.set("should have thrown an exception");
      } catch (InterruptedException | ExecutionException e) {
        returned.set(e.getCause());
      }
    };


    waitAndSet();

    assertThat(returned.get(), equalTo(failWith));
  }


  public void waitAndSet() throws Exception {
    assertFalse(future.isDone());

    Thread getThread = new Thread(getCode);
    getThread.start();

    // give the other thread time to be waiting
    waitFor(getThread, State.WAITING);

    assertFalse(future.isDone());

    if (failWith instanceof Problem) {
      future.markFailed((Problem) failWith);
    } else if (failWith instanceof Throwable) {
      future.markFailed((Throwable) failWith);
    } else {
      future.markComplete();
    }

    assertTrue(future.isDone());

    // give the other thread time to be waiting
    waitFor(getThread, State.TERMINATED);
  }

  private void waitFor(Thread thread, State desiredState) throws InterruptedException {
    while (thread.getState() != desiredState) {
      Thread.sleep(10);
    }
  }

}
