/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.sched;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.IntStream;

import org.junit.Test;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;

public class CombinedPageBufferTest {

  Struct struct = Types.INTEGER.asStruct();
  int pageCapacity = 10000;
  LinkedListBuffer childBuffer1 = new LinkedListBuffer(1, pageCapacity);
  LinkedListBuffer childBuffer2 = new LinkedListBuffer(1, pageCapacity);
  CombinedPageBuffer testBuffer = new CombinedPageBuffer(childBuffer1, childBuffer2);

  @Test
  public void canReportCombinedSize() {
    // buffer should report the combined size of child1 + child2
    assertTrue(testBuffer.isEmpty());
    assertEquals(0, testBuffer.size());
    assertEquals(0, testBuffer.numTuplesRead());

    addPage(childBuffer1, 1);
    assertFalse(testBuffer.isEmpty());
    assertEquals(1, testBuffer.size());

    assertThat(read(), is(1));
    assertTrue(testBuffer.isEmpty());
    assertEquals(0, testBuffer.size());
    assertEquals(1, testBuffer.numTuplesRead());

    addPage(childBuffer2, 1);
    assertFalse(testBuffer.isEmpty());
    assertEquals(1, testBuffer.size());

    addPage(childBuffer1, 2);
    addPage(childBuffer1, 3);
    assertEquals(3, testBuffer.size());

    assertThat(readAll(), containsInAnyOrder(1, 2, 3));
    assertTrue(testBuffer.isEmpty());
    assertEquals(0, testBuffer.size());
    assertEquals(4, testBuffer.numTuplesRead());
  }

  @Test
  public void canReportCompleteState() {
    // buffer should only be complete once all child buffers are complete
    assertFalse(testBuffer.isComplete());
    childBuffer1.markComplete();
    assertFalse(testBuffer.isComplete());
    childBuffer2.markComplete();
    assertTrue(testBuffer.isComplete());
  }

  @Test
  public void canReadFairlyFromAllBuffers() {
    // buffer shouldn't cause blockages because it favours one child over another
    IntStream.range(0, 10000).forEach(n -> addPage(childBuffer1, n));
    IntStream.range(0, 10000).forEach(n -> addPage(childBuffer2, n));
    assertThat(testBuffer.size(), is(20000));

    // read half the values and check one child hasn't been starved
    // (allow a healthy margin of error, as the child buffer to read is randomly picked)
    List<Integer> values = readUpTo(10000);
    assertThat(testBuffer.size(), is(10000));

    assertThat(Collections.max(values), lessThanOrEqualTo(7500));
    assertThat(childBuffer1.size(), lessThanOrEqualTo(7500));
    assertThat(childBuffer2.size(), lessThanOrEqualTo(7500));

    values = readAll();
    assertThat(Collections.max(values), is(9999));
    assertThat(childBuffer1.size(), is(0));
    assertThat(childBuffer2.size(), is(0));
    assertThat(testBuffer.size(), is(0));
  }

  public Integer read() {
    return ((Long) testBuffer.read().next().fetch("value")).intValue();
  }

  public List<Integer> readAll() {
    return readUpTo(Integer.MAX_VALUE);
  }

  public List<Integer> readUpTo(int numTimes) {
    List<Integer> read = new ArrayList<>();
    int i = 0;
    while (!testBuffer.isEmpty() && i < numTimes) {
      read.add(read());
      i++;
    }
    return read;
  }

  public void addPage(LinkedListBuffer buffer, long value) {
    buffer.add(pageOf(buffer, value, 1));
  }

  public Page pageOf(LinkedListBuffer buffer, long value, int repeated) {
    Page page = buffer.newPage();
    int i;
    for (i = 0; i < repeated; i++) {
      page.add(Tuple.ofValues(struct, value));
    }
    return page;
  }
}
