/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.sched;

import static org.mockito.Mockito.*;

import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.junit.Before;

import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.pipeline.Realized;
import nz.org.riskscape.engine.pipeline.RealizedPipeline;
import nz.org.riskscape.engine.pipeline.RealizedStep;
import nz.org.riskscape.engine.pipeline.TestPipelineJobContext;
import nz.org.riskscape.engine.relation.ListRelation;
import nz.org.riskscape.engine.relation.TupleIterator;
import nz.org.riskscape.engine.task.TaskSpec;
import nz.org.riskscape.engine.task.WorkerTask;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.ProblemException;

/**
 * Base class for task tests - is a little kludgy to support the original style of the {@link ChainTaskTest} but we
 * should evolve it over time to de-kludge
 */
@RequiredArgsConstructor
public class BaseTaskTest<T extends WorkerTask> extends ProjectTest {

  public final Class<T> taskClass;

  public Struct struct = Types.INTEGER.asStruct();
  public ListRelation longs = new ListRelation(struct, longs(0, 5));
  public LinkedListBuffer input = new LinkedListBuffer();
  public LinkedListBuffer output = new LinkedListBuffer();
  public T workerTask;
  public TaskSpec spec;

  RealizedPipeline pipeline = mock(RealizedPipeline.class);
  TestPipelineJobContext context = new TestPipelineJobContext(pipeline);

  @Before
  public void setup() {
    when(pipeline.getContext()).thenReturn(executionContext);
  }

  public List<Tuple> longs(int start, int stop) {
    return IntStream.rangeClosed(start, stop)
        .mapToObj(i -> Tuple.ofValues(struct, Long.valueOf(i)))
        .collect(Collectors.toList());
  }

  public LinkedListBuffer newBuffer(int pageSize, int pageCapacity) {
    return new LinkedListBuffer(pageSize, pageCapacity);
  }

  public TupleIterator source() {
    return longs.iterator();
  }

  public void fill(LinkedListBuffer inputBuffer, Iterator<Tuple> tuples) {
    fill(inputBuffer, tuples, true);
  }

  /**
   * populate the given buffer with the given tuples
   * @param markComplete if true call markComplete at the end
   */
  public void fill(LinkedListBuffer inputBuffer, Iterator<Tuple> tuples, boolean markComplete) {
    Page page = inputBuffer.newPage();

    while (tuples.hasNext()) {
      if (page.isFull()) {
        inputBuffer.add(page);
        page = inputBuffer.newPage();
      }
      page.add(tuples.next());
    }
    inputBuffer.add(page);

    if (markComplete) {
      inputBuffer.markComplete();
    }
  }

  public void fillInput() {
    fill(input, source());
  }

  public T createTask(RealizedStep... steps) {
    spec = new TaskSpec(taskClass, Arrays.asList(steps), input, output, true, context);
    try {
      workerTask = taskClass.cast(spec.newWorkerTask(null));
      return workerTask;
    } catch (ProblemException e) {
      throw new RuntimeException(e);
    }
  }

  public RealizedStep newStep(String name, Realized r) {
    return RealizedStep.named(name).withResult(r);
  }


}
