/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.rl.agg;

import static org.hamcrest.Matchers.*;

import org.junit.Test;

import static org.junit.Assert.*;

import java.util.Arrays;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.rl.RealizedExpression;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;

public class ModeAggregationFunctionTest extends BaseAggregationFunctionTest {

  @Test
  public void findsModeOfIntegers() {
    Struct inputType = norm(Types.INTEGER.asStruct());
    realized = realizeAggregate(inputType, parse("mode(value)")).get();

    assertThat(realized.getResultType(), is(Types.INTEGER));

    assertThat(this.process(), is(nullValue()));

    tuples = tuples(inputType, "[3]", "[2]", "[3]", "[4]", "[3]");
    assertThat(process(), is(3L));
  }

  @Test
  public void findsModeOfIntegersPickOneMostObserverd() {
    Struct inputType = norm(Types.INTEGER.asStruct());
    realized = realizeAggregate(inputType, parse("mode(value)")).get();

    assertThat(realized.getResultType(), is(Types.INTEGER));

    assertThat(this.process(), is(nullValue()));

    tuples = tuples(inputType, "[3]", "[2]", "[3]", "[4]", "[2]");
    assertThat(process(), oneOf(2L, 3L));
  }

  @Test
  public void findsModeOfText() {
    Struct inputType = norm(Struct.of("value", Nullable.TEXT));
    realized = realizeAggregate(inputType, parse("mode(value)")).get();

    assertThat(realized.getResultType(), is(Nullable.TEXT));

    assertThat(this.process(), is(nullValue()));

    tuples = tuples(inputType, "['cat']", "['dog']", "['cat']", "['bird']");
    assertThat(process(), is("cat"));
  }

  @Test
  public void findsModeOfStructTypes() {
    Struct type = Struct.of("name", Types.TEXT);
    Struct inputType = norm(Struct.of("value", type));

    realized = realizeAggregate(inputType, parse("mode(value)")).get();

    assertThat(realized.getResultType(), is(type));
    assertThat(this.process(), is(nullValue()));

    tuples = tuples(inputType, "[{name: 'bob'}]", "[{name: 'mary'}]", "[{name: 'bob'}]");
    assertThat(process(), is(Tuple.ofValues(type, "bob")));
  }

  @Test
  public void modeDoesNotCountNullValues() {
    Struct inputType = norm(Struct.of("value", Nullable.TEXT));
    realized = realizeAggregate(inputType, parse("mode(value)")).get();

    assertThat(realized.getResultType(), is(Nullable.TEXT));

    assertThat(this.process(), is(nullValue()));

    tuples = tuples(inputType, "[]", "['cat']", "[]");
    assertThat(process(), is("cat"));
  }

  @Test
  public void modeCalledAsFunction() {
    RSList listType = RSList.create(Types.INTEGER);
    Struct inputType = Struct.of("items", listType);
    RealizedExpression rExpr = realizationContext.getExpressionRealizer()
        .realize(inputType, parse("mode(items)")).get();

    Type resultType = Types.INTEGER;
    // result type is nullable because the list could be empty
    assertThat(rExpr.getResultType(), is(Nullable.of(resultType)));

    assertThat(rExpr.evaluate(Tuple.ofValues(inputType, Arrays.asList())), is(nullValue()));

    assertThat(rExpr.evaluate(Tuple.ofValues(inputType, Arrays.asList(3L, 4L, 3L))), is(3L));
  }

}
