/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.rl.agg;

import static nz.org.riskscape.engine.Matchers.*;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Arrays;

import org.junit.Test;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.rl.RealizedExpression;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;

@SuppressWarnings("unchecked")
public class MedianAggregationFunctionTest extends BaseAggregationFunctionTest {

  @Test
  public void doesMedianAggregationOnIntegers() {
    Struct inputType = norm(Types.INTEGER.asStruct());
    realized = realizeAggregate(inputType, parse("median(value)")).get();

    Type resultType = Types.FLOATING;
    assertThat(realized.getResultType(), is(resultType));

    assertThat(this.process(), is(nullValue()));

    tuples = tuples(inputType, "[4]", "[1]", "[9]", "[5]");
    assertThat(process(), is(4.5D));
  }

  @Test
  public void doesMedianAggregationOnFloating() {
    Struct inputType = norm(Types.FLOATING.asStruct());
    realized = realizeAggregate(inputType, parse("median(value)")).get();

    Type resultType = Types.FLOATING;
    assertThat(realized.getResultType(), is(resultType));

    assertThat(this.process(), is(nullValue()));

    tuples = tuples(inputType, "[0.4]", "[0.1]", "[0.9]", "[0.5]");
    assertThat(process(), is(0.45D));
  }

  @Test
  public void nullValuesDoNotAffectResult() {
    Struct inputType = norm(Nullable.INTEGER.asStruct());
    realized = realizeAggregate(inputType, parse("median(value)")).get();

    Type resultType = Types.FLOATING;
    assertThat(realized.getResultType(), is(Nullable.of(resultType)));

    assertThat(this.process(), is(nullValue()));

    tuples = tuples(inputType, "[4]", "[]", "[1]", "[]", "[9]", "[5]");
    assertThat(process(), is(4.5D));
  }

  @Test
  public void medianCalledAsFunction() {
    RSList listType = RSList.create(Types.INTEGER);
    Struct inputType = Struct.of("items", listType);
    RealizedExpression rExpr = realizationContext.getExpressionRealizer()
        .realize(inputType, parse("median(items)")).get();

    Type resultType = Types.FLOATING;
    // result type is nullable because the list could be empty
    assertThat(rExpr.getResultType(), is(Nullable.of(resultType)));

    assertThat(rExpr.evaluate(Tuple.ofValues(inputType, Arrays.asList())), is(nullValue()));

    assertThat(rExpr.evaluate(Tuple.ofValues(inputType, Arrays.asList(3L, 4L))), is(3.5D));
  }

  @Test
  public void errorWhenValueNotNumeric() {
    Struct inputType = norm(Types.TEXT.asStruct());
    assertThat(realizeAggregate(inputType, parse("median(value)")), failedResult(hasAncestorProblem(
        equalIgnoringChildren(
            ListCollecting.LocalProblems.get().valueNotNumeric("median", Types.TEXT))
        )));
  }
}
