/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.rl;

import static nz.org.riskscape.engine.Matchers.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.StringFunctions;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.rl.ast.ExpressionProblems;

public class StringFunctionsTest extends BaseExpressionRealizerTest {

  @Before
  public void before() {
    this.project.getFunctionSet().addAll(StringFunctions.FUNCTIONS);
    this.project.getFunctionSet().insertFirst(DefaultOperators.INSTANCE);
  }

  @Test
  public void canStrAnything() throws Exception {
    assertEquals("1", evaluate("str(1)", null));
    assertEquals("1.0", evaluate("str(1.0)", null));
    assertEquals("[foo, bar, baz]", evaluate("str(['foo', 'bar', 'baz'])", null));
  }

  @Test
  public void strOnNothingGivesEmptyString() throws Exception {
    this.inputStruct = Struct.of("foo", Nullable.TEXT);
    assertEquals("", evaluate("str(foo)", new Tuple((Struct) inputStruct)));
  }

  @Test
  public void canConcatenateStrings() throws Exception {
    assertEquals("foobar", evaluate("'foo' + 'bar'", null));
    assertEquals("foo1", evaluate("'foo' + str(1)", null));
  }

  @Test
  public void strPlusOperatorRequiresStrings() throws Exception {
    realize(inputStruct, parse("'foo' + 1"));
    assertThat(realizationProblems, contains(isError(ExpressionProblems.class, "noSuchOperatorFunction")));
  }

  @Test
  public void stringLength_SimpleText() {
    Struct type = Struct.of("foo", Types.TEXT);
    assertEquals(0L, evaluate("str_length(foo)", Tuple.ofValues(type, "")));
    assertEquals(1L, evaluate("str_length(foo)", Tuple.ofValues(type, "1")));
    assertEquals(8L, evaluate("str_length(foo)", Tuple.ofValues(type, "12345678")));
    assertEquals(Types.INTEGER, realized.getResultType());
  }

  @Test
  public void stringLength_NullableString() {
    Struct type = Struct.of("foo", Nullable.TEXT);
    assertNull(evaluate("str_length(foo)", Tuple.ofValues(type)));
    assertEquals(0L, evaluate("str_length(foo)", Tuple.ofValues(type, "")));
    assertEquals(1L, evaluate("str_length(foo)", Tuple.ofValues(type, "1")));
    assertEquals(8L, evaluate("str_length(foo)", Tuple.ofValues(type, "12345678")));
    assertEquals(Nullable.INTEGER, realized.getResultType());
  }

  @Test
  public void stringLength_NullableStruct() {
    Struct wrappedType = Struct.of("bar", Nullable.TEXT);
    Struct type = Struct.of("foo", Nullable.of(wrappedType));
    assertNull(evaluate("str_length(foo.bar)", Tuple.ofValues(type)));
    assertEquals(0L, evaluate("str_length(foo.bar)", Tuple.ofValues(type, Tuple.ofValues(wrappedType, ""))));
    assertEquals(1L, evaluate("str_length(foo.bar)", Tuple.ofValues(type, Tuple.ofValues(wrappedType, "1"))));
    assertEquals(8L, evaluate("str_length(foo.bar)", Tuple.ofValues(type, Tuple.ofValues(wrappedType, "12345678"))));
    assertEquals(Nullable.INTEGER, realized.getResultType());
  }

  @Test
  public void stringStartsWith() {
    this.inputStruct = Struct.of("foo", Nullable.TEXT, "bar", Nullable.TEXT);

    assertThat(evaluate("starts_with('assets', 'asset')", null), is(true));
    assertThat(evaluate("starts_with('assets', 'hazard')", null), is(false));

    assertThat(evaluate("starts_with(foo, bar)", Tuple.ofValues((Struct)inputStruct)), is(nullValue()));
    assertThat(evaluate("starts_with(foo, bar)", Tuple.ofValues((Struct)inputStruct, "asset")), is(nullValue()));
    assertThat(evaluate("starts_with(foo, bar)", Tuple.ofValues((Struct)inputStruct, null, "asset")), is(nullValue()));

    Tuple input = Tuple.ofValues((Struct)inputStruct, "asset");
    assertThat(evaluate("starts_with(foo, 'asset')", input), is(true));
    assertThat(evaluate("starts_with(foo, 'hazard')", input), is(false));

    input = Tuple.ofValues((Struct)inputStruct, "assets", "asset");
    assertThat(evaluate("starts_with(foo, bar)", input), is(true));
    assertThat(evaluate("starts_with(bar, foo)", input), is(false));
  }

  @Test
  public void stringEndsWith() {
    this.inputStruct = Struct.of("foo", Nullable.TEXT, "bar", Nullable.TEXT);

    assertThat(evaluate("ends_with('class', 'ass')", null), is(true));
    assertThat(evaluate("ends_with('assets', 'ass')", null), is(false));

    // if either argument is null, we get a null result
    assertThat(evaluate("ends_with(foo, bar)", Tuple.ofValues((Struct) inputStruct)), is(nullValue()));
    assertThat(evaluate("ends_with(foo, bar)", Tuple.ofValues((Struct) inputStruct, "asset")), is(nullValue()));
    assertThat(evaluate("ends_with(foo, bar)", Tuple.ofValues((Struct) inputStruct, null, "asset")), is(nullValue()));

    Tuple input = Tuple.ofValues((Struct) inputStruct, "class");
    assertThat(evaluate("ends_with(foo, 'ass')", input), is(true));
    assertThat(evaluate("ends_with(foo, 'asset')", input), is(false));
    assertThat(evaluate("ends_with(foo, 'ASS')", input), is(false));

    input = Tuple.ofValues((Struct) inputStruct, "class", "ass");
    assertThat(evaluate("ends_with(foo, bar)", input), is(true));
    assertThat(evaluate("ends_with(bar, foo)", input), is(false));
  }

  @Test
  public void stringFind() {
    this.inputStruct = Struct.of("foo", Nullable.TEXT, "bar", Nullable.TEXT);

    assertThat(evaluate("str_find('class', 'ass')", null), is(true));
    assertThat(evaluate("str_find('assets', 'ass')", null), is(true));
    assertThat(evaluate("str_find('assets', 'set')", null), is(true));
    assertThat(evaluate("str_find('assets', 'assets')", null), is(true));
    assertThat(evaluate("str_find('assets', 'assess')", null), is(false));
    assertThat(evaluate("str_find('assets', 'ASSET')", null), is(false));

    // if either argument is null, we get a null result
    assertThat(evaluate("str_find(foo, bar)", Tuple.ofValues((Struct) inputStruct)), is(nullValue()));
    assertThat(evaluate("str_find(foo, bar)", Tuple.ofValues((Struct) inputStruct, "asset")), is(nullValue()));
    assertThat(evaluate("str_find(foo, bar)", Tuple.ofValues((Struct) inputStruct, null, "asset")), is(nullValue()));

    Tuple input = Tuple.ofValues((Struct) inputStruct, "assets");
    assertThat(evaluate("str_find(foo, 'ass')", input), is(true));
    assertThat(evaluate("str_find(foo, 'set')", input), is(true));
    assertThat(evaluate("str_find(foo, 'x')", input), is(false));

    input = Tuple.ofValues((Struct) inputStruct, "class", "ass");
    assertThat(evaluate("str_find(foo, bar)", input), is(true));
    assertThat(evaluate("str_find(bar, foo)", input), is(false));
  }
}
