/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.rl;

import static org.hamcrest.Matchers.*;

import org.junit.Test;

import static org.junit.Assert.*;

import static nz.org.riskscape.engine.Matchers.*;

import org.junit.Before;

import com.google.common.collect.Range;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;


public class GetAttributeTest extends BaseExpressionRealizerTest {

  @Before
  public void setup() {
    project.getFunctionSet().add(LanguageFunctions.GET_ATTR);
    project.getFunctionSet().addAll(LogicFunctions.LOGIC_FUNCTIONS);
  }


  @Test
  public void returnsValueFromAttribute() {
    assertThat(evaluate("get_attr({foo: 'bar'}, attribute: 'foo')", Tuple.EMPTY_TUPLE), is("bar"));
    assertThat(realized.getResultType(), is(Types.TEXT));
  }

  @Test
  public void returnsDefaultValueWhenAttributeNotFount() {
    assertThat(evaluate("get_attr({foo: 'bar'}, attribute: 'baz', default: 33)", Tuple.EMPTY_TUPLE), is(33L));
    assertThat(realized.getResultType(), is(Types.INTEGER));
  }

  @Test
  public void defaultCanBeDynamic() {
    // users may want to chain the get_attr to hunt down the best option
    assertThat(evaluate("get_attr(value, attribute: 'foo', default: get_attr(value, 'bar'))",
        tuple("{value: {bar: 33}}")), is(33L));
    assertThat(realized.getResultType(), is(Types.INTEGER));
  }

  @Test
  public void returnsNullWhenAttributeNotInStruct() {
    assertThat(evaluate("get_attr({bar: 33}, attribute: 'foo')", Tuple.EMPTY_TUPLE), nullValue());
    assertThat(realized.getResultType(), is(Nullable.ANYTHING));
  }

  @Test
  public void failsWrongNumberOfArgs() {
    assertThat(realizeOnly("get_attr({foo: 'bar'})", Struct.EMPTY_STRUCT), failedResult(
        hasAncestorProblem(is(ArgsProblems.get().wrongNumber(Range.closed(2, 3), 1)))
    ));

    assertThat(realizeOnly("get_attr({foo: 'bar'}, 'foo', 'bar', 'baz')", Struct.EMPTY_STRUCT), failedResult(
        hasAncestorProblem(is(ArgsProblems.get().wrongNumber(Range.closed(2, 3), 4)))
    ));
  }

  @Test
  public void failsWhenValueArgNotStructType() {
    assertThat(realizeOnly("get_attr(10, 'foo')", Struct.EMPTY_STRUCT), failedResult(
        hasAncestorProblem(is(ArgsProblems.mismatch(GetAttribute.ARGUMENTS.get("struct"), Types.INTEGER)))
    ));
  }

  @Test
  public void failsWhenAttributeArgNotTextType() {
    assertThat(realizeOnly("get_attr({foo: 'bar'}, 10)", Struct.EMPTY_STRUCT), failedResult(
        hasAncestorProblem(is(ArgsProblems.mismatch(GetAttribute.ARGUMENTS.get("attribute"), Types.INTEGER)))
    ));
  }
}
