/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.rl;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Optional;

import org.junit.Test;

import nz.org.riskscape.rl.ExpressionParser;
import nz.org.riskscape.rl.ast.Expression;
import nz.org.riskscape.rl.ast.FunctionCall;

public class FunctionCallPrototypeTest {

  ExpressionParser parser = new ExpressionParser();

  @Test
  public void canUpdateValueInFunctionWithNoArgs() {
    assertThat(buildWithArgs("bob()", "value").toSource(),
        is("bob(value)"));
  }

  @Test
  public void canUpdateValuesInFunctionWithNoArgs() {
    assertThat(buildWithArgs("bob()", "a1", "a2", "a3").toSource(),
        is("bob(a1, a2, a3)"));
  }

  @Test
  public void canUpdateValueInFunctionWithValueArg() {
    assertThat(buildWithArgs("bob(default_value)", "value").toSource(),
        is("bob(value)"));
  }

  @Test
  public void canUpdateValueInFunctionWithExtraArgs() {
    assertThat(buildWithArgs("bob(default_value, 10)", "value").toSource(),
        is("bob(value, 10)"));
  }

  @Test
  public void canUpdateValueInFunctionWithExtraNamedArgs() {
    assertThat(buildWithArgs("bob(default_value, extra: 10)", "value").toSource(),
        is("bob(value, extra: 10)"));
  }

  @Test
  public void canSetNamedArg() {
    FunctionCallPrototype prototype = toFunctionPrototype("foo(bar, baz)");
    prototype.setArgument(1, Optional.of("second"), "2");

    assertThat(prototype.build().toSource(), is("foo(bar, second: 2)"));
  }

  @Test
  public void canAddArgs() {
    FunctionCallPrototype prototype = toFunctionPrototype("foo(bar)");

    prototype.addArgument(Optional.empty(), "100");
    prototype.addArgument(Optional.of("baz"), "50");

    assertThat(prototype.build().toSource(), is("foo(bar, 100, baz: 50)"));
  }

  @Test
  public void canSetArgsInAnyOrderIfTheyAlreadyExist() {
    FunctionCallPrototype prototype = toFunctionPrototype("foo(bar, baz)");
    // Not recommended to set args in random order, but if they already exist it isn't a problem
    prototype.setArgument(1, "second");
    prototype.setArgument(0, "first");

    assertThat(prototype.build().toSource(), is("foo(first, second)"));
  }

  FunctionCallPrototype toFunctionPrototype(String expression) {
    return new FunctionCallPrototype(parseFunction(expression));
  }

  FunctionCall parseFunction(String expression) {
    return parser.parse(expression)
        .isA(FunctionCall.class)
        .orElseThrow(() -> new RuntimeException());
  }

  Expression parse(String expression) {
    return parser.parse(expression);
  }

  private FunctionCall buildWithArgs(String startWith, String... arguments) {
    FunctionCallPrototype prototype = toFunctionPrototype(startWith);
    if (arguments.length == 1) {
      // If there is only one argument we use setFirstArgument to ensure it is tested to.
      prototype.setFirstArgument(arguments[0]);
    } else {
      for (int i = 0; i < arguments.length; i++) {
        prototype.setArgument(i, arguments[i]);
      }
    }
    return prototype.build();
  }
}
