/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.rl;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import org.junit.Test;
import org.locationtech.jts.geom.TopologyException;

import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.engine.bind.Parameter;
import nz.org.riskscape.engine.problem.GeneralProblems;

public class EvalExceptionTest extends BaseExpressionRealizerTest {

  @Test
  public void getRootCauseGivesTheFirstNonEvalExceptionFromNestedExceptions() {
    Throwable npe = new NullPointerException();
    Throwable riskscapeException = new RiskscapeException("foo", npe);
    RealizedExpression re = RealizedExpression.TRUE;
    Object input = "bar";
    EvalException inner = new EvalException(riskscapeException, re, input);
    EvalException outer = new EvalException(inner, re, input);

    assertSame(outer.getRootCause(), riskscapeException);
    assertSame(inner.getRootCause(), riskscapeException);
  }

  @Test
  public void exceptionMessageIsRenderedForJavaExceptions() {
    // Eval exceptions can wrap any throwable that comes their way. We want to to ensure that when
    // the eval exception gets rendered then it has some details on what went wrong.
    Throwable t = new RuntimeException("boom, boom");

    RealizedExpression re = RealizedExpression.TRUE;
    EvalException ex = new EvalException(t, re, "bar");
    String rendered = render(ex);
    assertThat(rendered, is("Failed to evaluate `true`\n"
        + "  - java.lang.RuntimeException: boom, boom"));
  }

  @Test
  public void problemIsRenderedIfAvailable() {
    Throwable t = new RiskscapeException(GeneralProblems.required("thing", Parameter.class));

    RealizedExpression re = RealizedExpression.TRUE;
    EvalException ex = new EvalException(t, re, "bar");
    String rendered = render(ex);
    assertThat(rendered, is("Failed to evaluate `true`\n"
        + "  - 'thing' parameter is required but was not found"));
  }

  @Test
  public void topologyExceptionsGetsATip() {
    Throwable t = new TopologyException("Mock topo exception");

    RealizedExpression re = RealizedExpression.TRUE;
    EvalException ex = new EvalException(t, re, "bar");
    String rendered = render(ex);
    assertThat(rendered, startsWith("Failed to evaluate `true`\n"
        + "  - This problem may mean that some geometry data is invalid. Try setting `validate-geometry = error`"));
  }

}
