/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.rl;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import org.junit.Test;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.StringFunctions;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.rl.ast.ExpressionProblems;

public class ConstantRealizingTest extends BaseExpressionRealizerTest {

  @Test
  public void willRealizeAnyConstantValue() throws Exception {
    assertEquals(1L, evaluateConstant("1"));
    assertEquals(1.1D, evaluateConstant("1.1"));
    assertEquals("foo", evaluateConstant("'foo'"));
  }

  @Test
  public void willRealizeTuples() {
    assertThat(evaluateConstant("{foo: 'bar'}"), is(Tuple.ofValues(Struct.of("foo", Types.TEXT), "bar")));

    // now with hypens in the name
    assertThat(evaluateConstant("{foo0-100: 'bar'}"),
        is(Tuple.ofValues(Struct.of("foo0-100", Types.TEXT), "bar")));
    assertThat(evaluateConstant("{'bar' as \"foo0-100\"}"),
        is(Tuple.ofValues(Struct.of("foo0-100", Types.TEXT), "bar")));
  }


  @Test
  public void failsIfReferencesAProperty() throws Exception {
    evaluateConstant("foo");
    assertThat(
      realizationProblems,
      contains(ExpressionProblems.get().constantRequired(expr))
    );
  }

  @Test
  public void willRealizeAConstantFunctionCall() throws Exception {
    project.getFunctionSet().addAll(StringFunctions.FUNCTIONS);

    assertEquals("1", evaluateConstant("str(1)"));
  }

  @Test
  public void failsIfFunctionCallArgReferencesAProperty() throws Exception {
    project.getFunctionSet().addAll(StringFunctions.FUNCTIONS);
    evaluateConstant("str(foo)");
    assertThat(
      realizationProblems,
      contains(ExpressionProblems.get().constantRequired(expr))
    );
  }
}
