/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.rl;

import static nz.org.riskscape.engine.Matchers.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.StringFunctions;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.problem.Problem.Severity;

@SuppressWarnings("unchecked")
public class BooleanExpressionSplitterTest extends ProjectTest {

  Struct inputType;

  public BooleanExpressionSplitterTest() {
    this.project.getFunctionSet().insertFirst(DefaultOperators.INSTANCE);
    this.project.getFunctionSet().addAll(StringFunctions.FUNCTIONS);
    this.project.getFunctionSet().addAll(LogicFunctions.LOGIC_FUNCTIONS);
  }

  @Before
  public void setupInputType() {
    inputType = Struct.of("integer", Types.INTEGER, "text", Types.TEXT, "ntext", Nullable.TEXT);
  }

  @Test
  public void canApplySimpleBooleanExpression() {
    ResultOrProblems<RealizedExpression> realized = splitAndRealize("text = 'dog'");
    assertThat(realized.hasProblems(), is(false));
    assertThat(realized.get().evaluate(Tuple.ofValues(inputType, 3L, "dog")), is(true));
    assertThat(realized.get().evaluate(Tuple.ofValues(inputType, 3L, "cat")), is(false));
  }

  @Test
  public void canApplyKnownPartsOfAndExpression() {
    ResultOrProblems<RealizedExpression> realized = splitAndRealize("(text = 'dog') AND (name = 'unknown')");
    assertThat(realized.hasProblems(), is(false));
    assertThat(realized.get().evaluate(Tuple.ofValues(inputType, 3L, "dog")), is(true));
    assertThat(realized.get().evaluate(Tuple.ofValues(inputType, 3L, "cat")), is(false));
  }

  @Test
  public void canApplyKnownPartsOfAndExpression1() {
    ResultOrProblems<RealizedExpression> realized =
        splitAndRealize("((text = 'dog') OR (integer = 3)) AND (name = 'unknown')");
    assertThat(realized.hasProblems(), is(false));
    assertThat(realized.get().evaluate(Tuple.ofValues(inputType, 3L, "dog")), is(true));
    assertThat(realized.get().evaluate(Tuple.ofValues(inputType, 3L, "cat")), is(true));
    assertThat(realized.get().evaluate(Tuple.ofValues(inputType, 3L, "bird")), is(true));
    assertThat(realized.get().evaluate(Tuple.ofValues(inputType, 2L, "bird")), is(false));

  }

  @Test
  public void canApplyExpressionWhenPropertyAccessesAreNestedInFunctionCalls() {
    ResultOrProblems<RealizedExpression> realized =
        splitAndRealize("integer = (str_length(text) / 2)");
    assertThat(realized.hasProblems(), is(false));

    assertThat(realized.get().evaluate(Tuple.ofValues(inputType, 2L, "bird")), is(true));
    assertThat(realized.get().evaluate(Tuple.ofValues(inputType, 1L, "bird")), is(false));
  }

  @Test
  public void canApplyExpressionWhenAllPropertyAccessesAreKnown() {
    ResultOrProblems<RealizedExpression> realized =
        splitAndRealize("text = ntext");
    assertThat(realized.hasProblems(), is(false));

    assertThat(realized.get().evaluate(Tuple.ofValues(inputType, 2L, "bird", "bird")), is(true));
    assertThat(realized.get().evaluate(Tuple.ofValues(inputType, 1L, "bird", "cat")), is(false));
  }

  @Test
  public void failsIfNoExpressionsCanBeApplied() {
    assertThat(splitAndRealize("name = 'unknown'"), failedResult(
        isProblem(Severity.ERROR,
            containsString("Could not apply 'name = 'unknown'' to type {integer=>Integer, text"))));

    assertThat(splitAndRealize("(text = 'dog') OR (name = 'unknown')"), failedResult(
        isProblem(Severity.ERROR, containsString(
            "Could not apply '(text = 'dog') OR (name = 'unknown')' to type {integer=>Integer, text"))));

    assertThat(splitAndRealize("text = unknown_attr"), failedResult(
        isProblem(Severity.ERROR, containsString(
            "Could not apply 'text = unknown_attr' to type {integer=>Integer, text"))));
  }

  @Test
  public void failsIfSecondLevelAttrUnknown() {
    inputType = Struct.of("foo", Struct.of("a", Types.TEXT, "b", Types.TEXT),
        "bar", Types.TEXT);

    assertThat(splitAndRealize("foo.c = 'what'"), failedResult(
        isProblem(Severity.ERROR,
            containsString("Could not apply 'foo.c = 'what'' to type {foo=>{a=>Text, b=>Text"))));
  }

  private ResultOrProblems<RealizedExpression> splitAndRealize(String expression) {
    return BooleanExpressionSplitter.splitAndRealize(expressionParser.parse(expression), inputType, expressionRealizer);
  }

}
