/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.rl;

import java.util.List;

import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ExpressionParser;
import nz.org.riskscape.rl.ast.Expression;
import nz.org.riskscape.rl.ast.FunctionCall;

/**
 * Base class for test cases that want to evaluate a pipeline expression
 */
public abstract class BaseExpressionRealizerTest extends ProjectTest {

  protected Type inputStruct = Struct.of();
  protected ExpressionParser parser = super.expressionParser;
  protected ExpressionRealizer realizer = super.expressionRealizer;
  protected Expression expr;
  protected List<Problem> realizationProblems;
  protected RealizedExpression realized;
  protected Object evaluated;

  protected ResultOrProblems<RealizedExpression> realizeOnly(String expression, Type inputType) {
    return realizer.realize(inputType, parse(expression));
  }

  protected RealizedExpression realize(Type inputType, Expression anExpr) {
    ResultOrProblems<RealizedExpression> realizedOr = realizer.realize(inputType, anExpr);
    realizationProblems = realizedOr.getProblems();
    realized = realizedOr.orElse(null);
    return realized;
  }

  protected Expression parse(String string) {
    return parser.parse(string);
  }

  protected Object evaluate(String expression, Tuple scope) {
    //Clear evaluated so a previous correct answer can't result in false positives.
    evaluated = null;
    expr = parse(expression);

    Type realizeAgainst = scope == null ? inputStruct : scope.getStruct();
    realized = realize(realizeAgainst, expr);
    if (realized != null) {
      evaluated = realized.evaluate(scope);
    }
    return evaluated;
  }


  protected Object evaluateConstant(String toParse) {
    evaluated = null;
    expr = parser.parse(toParse);

    ResultOrProblems<RealizedExpression> realizedOr = realizer.realizeConstant(expr);

    realized = realizedOr.orElse(null);
    realizationProblems = realizedOr.getProblems();

    if (realized != null) {
      evaluated = realized.evaluate(Tuple.EMPTY_TUPLE);
    }

    return evaluated;
  }

  protected RiskscapeFunction keywordArgFunction(ArgumentList list) {
    return new RiskscapeFunction() {

      @Override
      public Type getReturnType() {
        return RSList.create(Types.ANYTHING);
      }

      @Override
      public ArgumentList getArguments() {
        return list;
      }

      @Override
      public List<Type> getArgumentTypes() {
        return getTypesFromArguments();
      }

      @Override
      public Object call(List<Object> args) {
        return args;
      }
    };
  }

  protected FunctionCall.Argument getFunctionCallArg(int index) {
    return expr.isA(FunctionCall.class).map(fc -> fc.getArguments().get(index)).get();
  }
}
