/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.resource;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.net.URI;
import java.nio.file.Files;
import java.nio.file.Path;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.TemporaryDirectoryTestHelper;

public class UriHelperTest implements TemporaryDirectoryTestHelper {

  Path tmpDir;
  URI relativeTo;

  @Before
  public void setup() throws Exception {
    // by default, resolve everything relative to a temp dir
    tmpDir = createTempDirectory("UriHelperTest");
    relativeTo = tmpDir.toUri();
  }

  @After
  public void breakdown() throws Exception {
    remove(tmpDir);
  }

  private URI getUri(String location) {
    return UriHelper.uriFromLocation(location, relativeTo).get();
  }

  private URI getDirectoryUri(String location) {
    return UriHelper.directoryUriFromLocation(location, relativeTo).get();
  }

  @Test
  public void canGetRelativeUri() {
    assertThat(getUri("path/to/file").toString(), is(relativeTo.toString() + "path/to/file"));
  }

  @Test
  public void spacesAreEscaped() {
    URI hasSpaces = getUri("has many spaces.csv");
    assertThat(hasSpaces.toString(), is(relativeTo.toString() + "has%20many%20spaces.csv"));
    assertThat(hasSpaces.getPath(), is(relativeTo.getPath() + "has many spaces.csv"));
  }

  @Test
  public void absoluteLocationWillIgnoreRelativeTo() throws Exception {
    Path tmpDir2 = createTempDirectory("UriHelperTest2");
    Path path = tmpDir2.resolve("path");

    URI uri = getUri(path.toAbsolutePath().toString());
    assertEquals(uri, path.toUri());
    assertThat(uri.toString(), not(containsString(relativeTo.toString())));
  }

  @Test
  public void canGetRelativeDirectoryUriRegardlessOfExistence() throws Exception {
    // the "/" on the input location indicates a directory, so we expect the same result
    // regardless of whether the directory actually exists or not
    assertThat(getUri("path/").toString(), is(relativeTo.toString() + "path/"));
    Files.createDirectory(tmpDir.resolve("path"));
    assertThat(getUri("path/").toString(), is(relativeTo.toString() + "path/"));

    // when the location exists *and* it's a directory, we always expect to get a trailing slash
    // back, even if the input location doesn't contain one
    assertThat(getUri("path").toString(), is(relativeTo.toString() + "path/"));
  }

  @Test
  public void canGetRelativeFileUriRegardlessOfExistence() throws Exception {
    // there's no trailing "/" on the input location to indicate this is a directory,
    // so we don't expect the UriHelper to jump to any conclusions
    assertThat(getUri("path").toString(), is(relativeTo.toString() + "path"));
    Files.createFile(tmpDir.resolve("path"));
    assertThat(getUri("path").toString(), is(relativeTo.toString() + "path"));
  }

  @Test
  public void canGetRelativeDirectoryUri() throws Exception {
    assertThat(getDirectoryUri("path/").toString(), is(relativeTo.toString() + "path/"));

    // if users don't put on the trailing slash then it should be added for them
    assertThat(getDirectoryUri("path").toString(), is(relativeTo.toString() + "path/"));
  }

  @Test
  public void spacesGetEncoded() throws Exception {
    relativeTo = URI.create("https://riskscape.nz/dav/");

    URI coolFile = getUri("foo directory/cool file.csv");
    assertThat(coolFile.getRawPath(), is("/dav/foo%20directory/cool%20file.csv"));
    assertThat(coolFile.getPath(), is("/dav/foo directory/cool file.csv"));
  }

  @Test
  public void canDetectLocationWithScheme() {
    assertTrue(UriHelper.hasUriScheme("file:///path/to/file"));
    assertTrue(UriHelper.hasUriScheme("https://host/file"));

    assertFalse(UriHelper.hasUriScheme("path/to/file"));
    assertFalse(UriHelper.hasUriScheme("/path/to/file"));
    // a windows file path shouldn't be confused with a uri scheme
    assertFalse(UriHelper.hasUriScheme("C:\\path\to\file"));
  }

}
