/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.resource;

import static nz.org.riskscape.engine.Assert.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;
import static org.mockito.ArgumentMatchers.*;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.*;

import java.net.URI;

import org.junit.Test;

public class ResourceFactoryTest {

  ResourceFactory factory;

  @Test
  public void errorIfNoLoaderCanLoad() throws Exception {
    factory = new DefaultResourceFactory();
    ResourceLoadingException ex = assertThrows(ResourceLoadingException.class,
        () -> factory.load(new URI("unknown://cannot.load.com"), new URI("file:///path")));
    assertThat(ex.getMessage(), containsString("Could not find a Resource Loader for 'unknown://cannot.load.com'"));
  }

  @Test
  public void willFindMatch() throws Exception {
    ResourceLoader mock1 = mock(ResourceLoader.class);
    when(mock1.canLoad(any())).thenReturn(true);
    when(mock1.load(any())).thenReturn(
        new StringResource("mock-resource-any", "the mock resource"));
    factory = new DefaultResourceFactory(mock1);

    Resource result = factory.load(new URI("mock://resource"), new URI("file:///path"));
    assertThat(result.getLocation().toString(), containsString("mock-resource-any"));
  }

  @Test
  public void willFindMatchWhenRelative() throws Exception {
    URI toMatch = new URI("file:///path/to/my/file");
    ResourceLoader mock1 = mock(ResourceLoader.class);
    when(mock1.canLoad(eq(toMatch))).thenReturn(true);
    when(mock1.load(eq(toMatch))).thenReturn(
        new StringResource("mock-resource-any", "the mock resource"));

    factory = new DefaultResourceFactory(mock1);
    Resource result = factory.load(new URI("my/file"), new URI("file:///path/to/"));
    assertThat(result.getLocation().toString(), containsString("mock-resource-any"));
  }

  @Test
  public void willFindFirstMatch() throws Exception {
    URI mockURI = new URI("mock://resource1");
    ResourceLoader mock1 = mock(ResourceLoader.class);
    when(mock1.canLoad(any())).thenReturn(true);
    when(mock1.load(any())).thenReturn(
        new StringResource("mock-any", "the mock resource"));

    ResourceLoader mock2 = mock(ResourceLoader.class);
    when(mock2.canLoad(eq(mockURI))).thenReturn(true);
    when(mock2.load(eq(mockURI))).thenReturn(
        new StringResource("mock-resource1", "the mock resource"));

    factory = new DefaultResourceFactory(mock1, mock2);
    Resource result = factory.load(new URI("mock://resource1"), new URI("file:///path"));
    assertThat(result.getLocation().toString(), containsString("mock-any"));
  }

}
