/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.relation;

import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.stream.Collectors;

import org.hamcrest.Matchers;
import org.junit.Test;

import nz.org.riskscape.engine.Assert;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;

public class TupleIteratorTest {

  @Test
  public void willHookCloseUpToOptionalRunnable() {
    Runnable mockRunnable = mock(Runnable.class);
    TupleIterator wrapped = TupleIterator.wrapped(Arrays.<Tuple>asList().iterator(), Optional.of(mockRunnable));
    wrapped.close();
    verify(mockRunnable).run();
  }

  @Test
  public void testCollectFunction() throws Exception {
    List<Tuple> tuples = Collections.singletonList(Tuple.EMPTY_TUPLE);
    Runnable mockRunnable = mock(Runnable.class);
    TupleIterator wrapped = TupleIterator.wrapped(tuples.iterator(), Optional.of(mockRunnable));
    assertEquals(tuples, wrapped.collect(Collectors.toList()));
    verify(mockRunnable).run();
  }

  @Test
  public void thereIsAnEmptyIterator() throws Exception {
    assertFalse(TupleIterator.EMPTY.hasNext());
    Assert.assertThrows(NoSuchElementException.class, () -> TupleIterator.EMPTY.next());
  }

  @Test
  public void canCreateASingleton() throws Exception {
    Tuple tuple = Tuple.of(Struct.of());
    TupleIterator iterator = TupleIterator.singleton(tuple);

    assertTrue(iterator.hasNext());
    assertSame(tuple, iterator.next());
    assertFalse(iterator.hasNext());
    Assert.assertThrows(NoSuchElementException.class, () -> iterator.next());
  }

  @Test
  public void canRemoveValuesDuringIteration() {
    Map<String, Long> values = new HashMap<>();
    values.put("foo", 10L);
    values.put("bar", 20L);

    Struct type = Struct.of("name", Types.TEXT, "value", Types.INTEGER);
    List<Tuple> mapped = new ArrayList<>();

    TupleIterator iterator = TupleIterator.wrappedAndMapped(values.entrySet().iterator(),
        entry -> Tuple.ofValues(type, entry.getKey(), entry.getValue()));

    assertTrue(iterator.hasNext());
    mapped.add(iterator.next());
    iterator.remove();
    assertEquals(1, values.size());

    assertTrue(iterator.hasNext());
    mapped.add(iterator.next());
    iterator.remove();
    assertEquals(0, values.size());

    assertThat(mapped, Matchers.containsInAnyOrder(
        Tuple.ofValues(type, "foo", 10L),
        Tuple.ofValues(type, "bar", 20L)
    ));
  }

}
