/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.relation;

import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.stream.Collectors;

import org.geotools.api.data.Query;
import org.geotools.data.simple.SimpleFeatureCollection;
import org.geotools.data.simple.SimpleFeatureIterator;
import org.geotools.api.data.SimpleFeatureSource;
import org.geotools.util.factory.Hints;
import org.geotools.feature.collection.SimpleFeatureIteratorImpl;
import org.junit.Before;
import org.junit.Test;
import org.geotools.api.feature.simple.SimpleFeature;
import org.geotools.api.feature.simple.SimpleFeatureType;
import org.geotools.api.filter.Filter;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

import com.google.common.collect.Lists;

import nz.org.riskscape.engine.Assert;
import nz.org.riskscape.engine.CrsHelper;
import nz.org.riskscape.engine.RiskscapeIOException;
import nz.org.riskscape.engine.SRIDSet;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.projection.NestMembersProjection;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;

public class PagingFeatureSourceRelationTest implements CrsHelper {

  int pageSize = 5;
  SimpleFeatureSource mockFeatureSource = mock(SimpleFeatureSource.class);
  Query expectedQuery = new Query();
  SimpleFeatureCollection mockFeatureCollection = mock(SimpleFeatureCollection.class);
  SimpleFeatureIterator mockFeatureIterator = mock(SimpleFeatureIterator.class);
  Struct type = Types.TEXT.asStruct().add("_fid", Types.TEXT);
  SRIDSet sridSet = new SRIDSet();
  PagingFeatureSourceRelation relation;

  SimpleFeatureCollection emptyFeatureCollection = mock(SimpleFeatureCollection.class);
  CoordinateReferenceSystem crs = nzMapGrid();


  @Before
  public void setup() {
    expectedQuery.setStartIndex(0);
    expectedQuery.setMaxFeatures(pageSize);
    Hints hints = new Hints();
    hints.put(Hints.JTS_GEOMETRY_FACTORY, sridSet.getGeometryFactory(crs));
    hints.put(Hints.FEATURE_2D, true);
    expectedQuery.setHints(hints);

    SimpleFeatureType featureType = mock(SimpleFeatureType.class);

    when(featureType.getCoordinateReferenceSystem()).thenReturn(crs);
    when(mockFeatureSource.getSchema()).thenReturn(featureType);

    SimpleFeatureIterator emptyIterator = featureIterator();
    when(emptyFeatureCollection.features()).thenReturn(emptyIterator);

    this.relation = new PagingFeatureSourceRelation(type, mockFeatureSource, sridSet,
        Filter.INCLUDE, crs, pageSize);
  }

  @Test
  public void anEmptyFeatureSourceReturnsNoResult() throws IOException {
    when(mockFeatureCollection.features()).thenReturn(mockFeatureIterator);
    when(mockFeatureIterator.hasNext()).thenReturn(false);
    when(mockFeatureIterator.next()).thenThrow(new NoSuchElementException());
    when(mockFeatureSource.getFeatures(expectedQuery)).thenReturn(mockFeatureCollection);

    TupleIterator iterator = relation.iterator();

    assertFalse(iterator.hasNext());
    Assert.assertThrows(NoSuchElementException.class, () -> iterator.next());
  }

  @Test
  public void aFeatureSourceWithLessThanAFullPageIsConsumedWithASingleFeatureSourceIterator() throws IOException {
    when(mockFeatureSource.getFeatures(expectedQuery)).thenReturn(mockFeatureCollection);
    SimpleFeatureIterator featureIterator = featureIterator("foo", "bar", "baz");
    when(mockFeatureCollection.features()).thenReturn(featureIterator);

    TupleIterator iterator = relation.iterator();

    assertTrue(iterator.hasNext());
    String foo = iterator.next().fetch("value");
    String bar = iterator.next().fetch("value");
    String baz = iterator.next().fetch("value");

    assertFalse(iterator.hasNext());
    assertEquals(Arrays.asList("foo", "bar", "baz"), Arrays.asList(foo, bar, baz));
    Assert.assertThrows(NoSuchElementException.class, () -> iterator.next());
  }

  @Test
  public void aPagingFeatureSourceRelationCanBeProjected() throws IOException {
    when(mockFeatureSource.getFeatures(expectedQuery)).thenReturn(mockFeatureCollection);
    SimpleFeatureIterator featureIterator = featureIterator("foo", "bar", "baz");
    when(mockFeatureCollection.features()).thenReturn(featureIterator);

    Relation projectedRelation = relation.project(new NestMembersProjection("child")).get();

    TupleIterator iterator = projectedRelation.iterator();

    assertTrue(iterator.hasNext());
    String foo = ((Tuple)iterator.next().fetch("child")).fetch("value");
    String bar = ((Tuple)iterator.next().fetch("child")).fetch("value");
    String baz = ((Tuple)iterator.next().fetch("child")).fetch("value");

    assertFalse(iterator.hasNext());
    assertEquals(Arrays.asList("foo", "bar", "baz"), Arrays.asList(foo, bar, baz));
    Assert.assertThrows(NoSuchElementException.class, () -> iterator.next());
  }


  @Test
  public void aFeatureSourceWithAPageWorthOfResultsIsUsedCompletely() throws IOException {
    Query secondQuery = new Query();
    secondQuery.setStartIndex(pageSize);
    secondQuery.setMaxFeatures(pageSize * 2);
    secondQuery.setHints(expectedQuery.getHints());

    List<String> expectedValues = Arrays.asList("foo1", "foo2", "foo3", "foo4", "foo5");

    when(mockFeatureSource.getFeatures(expectedQuery)).thenReturn(mockFeatureCollection);
    when(mockFeatureSource.getFeatures(secondQuery)).thenReturn(emptyFeatureCollection);
    SimpleFeatureIterator featureIterator = featureIterator(expectedValues);
    when(mockFeatureCollection.features()).thenReturn(featureIterator);

    TupleIterator iterator = relation.iterator();

    List<String> actualValues = Lists.newArrayList();
    while (iterator.hasNext()) {
      actualValues.add(iterator.next().fetch("value"));
    }

    assertEquals(expectedValues, actualValues);
    Assert.assertThrows(NoSuchElementException.class, () -> iterator.next());
  }


  @Test
  public void aFeatureSourceWithAPageAndABitWorthOfResultsIsUsedCompletely() throws IOException {
    Query secondQuery = new Query();
    secondQuery.setStartIndex(pageSize);
    secondQuery.setMaxFeatures(pageSize * 2);
    secondQuery.setHints(expectedQuery.getHints());

    List<String> firstExpectedValues = Arrays.asList("foo1", "foo2", "foo3", "foo4", "foo5");
    List<String> secondExpectedValues = Arrays.asList("foo6", "foo7", "foo8");

    when(mockFeatureSource.getFeatures(expectedQuery)).thenReturn(mockFeatureCollection);
    SimpleFeatureCollection secondFeatureCollection = mock(SimpleFeatureCollection.class);
    SimpleFeatureIterator secondIterator = featureIterator(secondExpectedValues);
    when(mockFeatureSource.getFeatures(secondQuery)).thenReturn(secondFeatureCollection);
    SimpleFeatureIterator featureIterator = featureIterator(firstExpectedValues);
    when(mockFeatureCollection.features()).thenReturn(featureIterator);
    when(secondFeatureCollection.features()).thenReturn(secondIterator);

    TupleIterator iterator = relation.iterator();

    List<String> actualValues = Lists.newArrayList();
    while (iterator.hasNext()) {
      actualValues.add(iterator.next().fetch("value"));
    }

    List<String> expectedValues = Lists.newArrayList();
    expectedValues.addAll(firstExpectedValues);
    expectedValues.addAll(secondExpectedValues);
    assertEquals(expectedValues, actualValues);
    Assert.assertThrows(NoSuchElementException.class, () -> iterator.next());
  }

  @Test
  public void ioExceptionsBubbleUp() throws IOException {
    when(mockFeatureSource.getFeatures(expectedQuery)).thenThrow(new IOException("bad juju"));
    SimpleFeatureIterator featureIterator = featureIterator("foo", "bar", "baz");
    when(mockFeatureCollection.features()).thenReturn(featureIterator);

    TupleIterator iterator = relation.iterator();
    Assert.assertThrows(RiskscapeIOException.class, () -> {
      iterator.next();
    });
  }

  @Test
  public void canApplyAFilterToTheUnderlyingFeatureSourceWithoutUpsettingThePaging() throws IOException {
    Filter someFilter = mock(Filter.class);
    expectedQuery.setFilter(someFilter);
    relation = new PagingFeatureSourceRelation(type, mockFeatureSource, sridSet, someFilter, crs, pageSize);

    List<String> expectedValues = Arrays.asList("foo1", "foo2", "foo3", "foo4");
    when(mockFeatureSource.getFeatures(expectedQuery)).thenReturn(mockFeatureCollection);
    SimpleFeatureIterator featureIterator = featureIterator(expectedValues);
    when(mockFeatureCollection.features()).thenReturn(featureIterator);

    TupleIterator iterator = relation.iterator();

    List<String> actualValues = Lists.newArrayList();
    while (iterator.hasNext()) {
      actualValues.add(iterator.next().fetch("value"));
    }

  }

  private SimpleFeatureIterator featureIterator(String... values) {
    return featureIterator(Arrays.asList(values));
  }

  private SimpleFeatureIterator featureIterator(List<String> listValues) {


    List<SimpleFeature> collection = listValues.stream().map((string) -> {
      SimpleFeature feature = mock(SimpleFeature.class);
      when(feature.getAttribute("value")).thenReturn(string);
      when(feature.getFeatureType()).thenReturn(mock(SimpleFeatureType.class));
      return feature;
    }).collect(Collectors.toList());

    return new SimpleFeatureIteratorImpl(collection);
  }
}
