/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.relation;

import nz.org.riskscape.engine.Tuple;

/**
 * Quick-n-dirty {@link TupleIterator} that suits being written to by one thread while being read from another.
 */
public class BufferedTupleIterator implements TupleIterator {

  private Object lock = new Object();

  private final TupleIterator source;

  private final Tuple[] buffer;

  private int size;

  private int position;

  private final int capacity;

  public BufferedTupleIterator(TupleIterator source, int bufferCapacity) {
    this.source = source;
    this.buffer = new Tuple[bufferCapacity];
    this.size = 0;
    this.position = 0;
    this.capacity = bufferCapacity;
  }

  @Override
  public boolean hasNext() {
    synchronized (lock) {
      if (size > 0) {
        return true;
      } else {
        return source.hasNext();
      }
    }
  }

  @Override
  public Tuple next() {
    synchronized (lock) {
      if (size > 0) {
        Tuple tuple = buffer[position++];
        position = position % capacity;
        size--;
        return tuple;
      } else {
        return source.next();
      }
    }
  }

  public boolean sourceHasNext() {
    synchronized (lock) {
      return source.hasNext();
    }
  }

  public boolean put() {
    boolean done = false;
    synchronized (lock) {

      while (!atCapacity()) {
        if (source.hasNext()) {
          Tuple tuple = source.next();
          done = true;
          buffer[(position + size++) % capacity] = tuple;
        } else {
          return done;
        }
      }
    }
    return done;
  }

  private boolean atCapacity() {
    synchronized (lock) {
      return size == capacity;
    }
  }

  public boolean hasBuffered() {
    synchronized (lock) {
      return size > 0;
    }
  }
}
