/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.query;


import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

import java.util.function.Supplier;

import org.geotools.geometry.jts.ReferencedEnvelope;
import org.junit.Test;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

import nz.org.riskscape.engine.Assert;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.query.TupleUtils.FindOption;
import nz.org.riskscape.engine.types.MultiGeom;
import nz.org.riskscape.engine.types.Referenced;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;

public class TupleUtilsTest {

  RuntimeException exception = new RuntimeException("hi mom");
  Supplier<RuntimeException> failedHandler = () -> exception;

  Struct oneGeom = Struct.of("foo", Types.TEXT, "bar", Types.GEOMETRY, "baz", Types.INTEGER);
  Struct onePointGeom = Struct.of("foo", Types.TEXT, "bar", Types.POINT, "baz", Types.INTEGER);
  Struct twoGeom = Struct.of("foo", Types.TEXT, "bar", Types.GEOMETRY, "baz", Types.GEOMETRY);
  Struct twoGeomMixedTypes = Struct.of("foo", Types.TEXT, "bar", Types.LINE, "baz", Types.GEOMETRY);
  Struct noGeom = Struct.of("foo", Types.TEXT, "bar", Types.DECIMAL, "baz", Types.INTEGER);

  @Test
  public void returnsTheGeometryMemberItFinds() {
    assertEquals("bar", TupleUtils.findGeometryMember(oneGeom, FindOption.ONE_REQUIRED, failedHandler).getKey());
  }

  @Test
  public void returnsThePointGeometryMemberItFinds() {
    assertEquals("bar", TupleUtils.findGeometryMember(onePointGeom, FindOption.ONE_REQUIRED, failedHandler).getKey());
  }

  @Test
  public void returnsTheReferenceGeometryMemberItFinds() {
    CoordinateReferenceSystem crs = mock(CoordinateReferenceSystem.class);
    ReferencedEnvelope envelope = mock(ReferencedEnvelope.class);

    Struct referencedGeom = Struct.of("value", Types.INTEGER,
        "referenced", Referenced.of(Types.GEOMETRY, crs, envelope));

    assertEquals("referenced",
        TupleUtils.findGeometryMember(referencedGeom, FindOption.ONE_REQUIRED, failedHandler).getKey());
  }

  @Test
  public void canReturnsTheFirstGeometryMemberItFinds() throws Exception {
    assertEquals("bar", TupleUtils.findGeometryMember(twoGeom, FindOption.ANY_REQUIRED, failedHandler).getKey());
  }

  @Test
  public void canReturnsTheFirstGeometryMemberItFindsFromMixedTypes() throws Exception {
    assertEquals("bar",
        TupleUtils.findGeometryMember(twoGeomMixedTypes, FindOption.ANY_REQUIRED, failedHandler).getKey());
  }

  @Test
  public void canBeSetToFailIfMoreThanOneGeometryMemberFound() throws Exception {
    assertSame(exception, Assert.assertThrows(RuntimeException.class, ()
        -> TupleUtils.findGeometryMember(twoGeom, FindOption.ONE_REQUIRED, failedHandler)));
  }

  @Test
  public void canBeSetToFailIfMoreThanOneGeometryMemberFoundFromMixedTypes() throws Exception {
    assertSame(exception, Assert.assertThrows(RuntimeException.class, ()
        -> TupleUtils.findGeometryMember(twoGeomMixedTypes, FindOption.ONE_REQUIRED, failedHandler)));
  }

  @Test
  public void canBeSetToFailIfNoGeometryMemberFound() throws Exception {
    assertSame(exception, Assert.assertThrows(RuntimeException.class, ()
        -> TupleUtils.findGeometryMember(noGeom, FindOption.ONE_REQUIRED, failedHandler)));

    assertSame(exception, Assert.assertThrows(RuntimeException.class, ()
        -> TupleUtils.findGeometryMember(noGeom, FindOption.ANY_REQUIRED, failedHandler)));

  }

  @Test
  public void detectsIfStructHasGeometryMember() {
    assertTrue(TupleUtils.hasGeometryMember(Struct.of("geom", Types.GEOMETRY)));
    assertTrue(TupleUtils.hasGeometryMember(Struct.of("geom", Types.GEOMETRY, "foo", Types.TEXT)));
    assertTrue(TupleUtils.hasGeometryMember(
        Struct.of("geom", Types.GEOMETRY, "foo", Types.TEXT, "geom2", Types.GEOMETRY)));

    // what about a geometry subtype
    assertTrue(TupleUtils.hasGeometryMember(Struct.of("geom", Types.POINT)));
    assertTrue(TupleUtils.hasGeometryMember(Struct.of("geom", Types.POLYGON, "foo", Types.TEXT)));
    assertTrue(TupleUtils.hasGeometryMember(
        Struct.of("geom", Types.LINE, "foo", Types.TEXT, "geom2", Types.GEOMETRY)));


    assertFalse(TupleUtils.hasGeometryMember(Struct.of("value", Types.INTEGER)));
  }

  @Test
  public void debugStringIsConcise() {
    Struct struct = Struct.of("foo", Types.TEXT, "bar", Types.TEXT);
    Tuple tuple = Tuple.ofValues(struct, "foo is where it is at", "whatever, bar is everything");

    assertEquals("{foo=foo is …, bar=whateve…}", TupleUtils.getContentSummary(tuple));

    struct = Struct.of("foo", Types.TEXT, "geom", Types.GEOMETRY, "point", Types.POINT).and("mgeom",
        MultiGeom.MULTI_LINE);
    tuple = Tuple.ofValues(struct, "foo is where it is at");

    assertEquals("{foo=foo is …}", TupleUtils.getContentSummary(tuple));
  }
}
