/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.projection;

import static nz.org.riskscape.engine.Matchers.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Arrays;

import org.junit.Test;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;

import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.rl.DefaultOperators;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.ExpressionProblems;

@SuppressWarnings("unchecked")
public class SetAttributeProjectionTest extends ProjectTest {

  Struct sourceType = Struct.of("foo", Types.TEXT).and("baz", Types.INTEGER);

  public SetAttributeProjectionTest() {
    project.getFunctionSet().insertFirst(DefaultOperators.INSTANCE);
  }

  @Test
  public void projectsToEquivalantTypeIfNoSetAttributes() {
    SetAttributeProjection projection = new SetAttributeProjection(Maps.newHashMap(), expressionRealizer);

    Projector projector = validate(projection.getProjectionFunction(sourceType), sourceType);
    Tuple input = Tuple.ofValues(sourceType, "bob", 30L);
    assertThat(projector.apply(input), is(input));
  }

  @Test
  public void canModifyAttributeValue() {
    SetAttributeProjection projection = new SetAttributeProjection(ImmutableMap.of(
        "baz", expressionParser.parse("baz * 2")
    ), expressionRealizer);

    Projector projector = validate(projection.getProjectionFunction(sourceType), sourceType);
    Tuple input = Tuple.ofValues(sourceType, "bob", 30L);
    assertThat(projector.apply(input), is(Tuple.ofValues(sourceType, "bob", 60L)));
  }

  @Test
  public void canModifyAttributeType() {
    SetAttributeProjection projection = new SetAttributeProjection(ImmutableMap.of(
        "baz", expressionParser.parse("baz / 2")
    ), expressionRealizer);

    Struct resultType = sourceType.addOrReplace("baz", Types.FLOATING);

    Projector projector = validate(projection.getProjectionFunction(sourceType), resultType);
    Tuple input = Tuple.ofValues(sourceType, "bob", 3L);
    assertThat(projector.apply(input), is(Tuple.ofValues(resultType, "bob", 1.5D)));
  }

  @Test
  public void caAddAttributeType() {
    SetAttributeProjection projection = new SetAttributeProjection(ImmutableMap.of(
        "halfBaz", expressionParser.parse("baz / 2")
    ), expressionRealizer);

    Struct resultType = sourceType.addOrReplace("halfBaz", Types.FLOATING);

    Projector projector = validate(projection.getProjectionFunction(sourceType), resultType);
    Tuple input = Tuple.ofValues(sourceType, "bob", 3L);
    assertThat(projector.apply(input), is(Tuple.ofValues(resultType, "bob", 3L, 1.5D)));
  }

  @Test
  public void errorWhenSetAttributeNotRealizable() {
    SetAttributeProjection projection = new SetAttributeProjection(ImmutableMap.of(
        "halfBaz", expressionParser.parse("ba / 2")
    ), expressionRealizer);
    assertThat(projection.getProjectionFunction(sourceType), failedResult(hasAncestorProblem(
        equalTo(ExpressionProblems.get().noSuchStructMember("ba", Arrays.asList("foo", "baz")))
    )));
  }

  protected Projector validate(ResultOrProblems<Projector> p, Struct expectedResultType) {
    assertThat(p.hasErrors(), is(false));
    Projector projector = p.get();
    assertThat(projector.getProjectedType(), is(expectedResultType));
    return projector;
  }

}
