/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.projection;

import static nz.org.riskscape.engine.Matchers.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import org.junit.Before;
import org.junit.Test;

import com.google.common.collect.Lists;

import nz.org.riskscape.engine.DummyFunction;
import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.rl.DefaultOperators;
import nz.org.riskscape.engine.rl.GeometryFunctions;
import nz.org.riskscape.engine.rl.LogicFunctions;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ExpressionParser;
import nz.org.riskscape.rl.ast.ExpressionProblems;
import nz.org.riskscape.rl.ast.StructDeclaration;

@SuppressWarnings("unchecked")
public class SelectProjectorTest extends ProjectTest {

  @Before
  public final void setupStep() {
    this.project.getFunctionSet().insertFirst(DefaultOperators.INSTANCE);
    GeometryFunctions geometryFunctions = new GeometryFunctions(engine);
    this.project.getFunctionSet().addAll(geometryFunctions.getPredicates());
    this.project.getFunctionSet().addAll(geometryFunctions.getFunctions());
    this.project.getFunctionSet().addAll(LogicFunctions.LOGIC_FUNCTIONS);

    this.project.getFunctionSet().add(
        new DummyFunction("dummy", Lists.newArrayList(Types.TEXT, Types.TEXT)) {
      @Override
      public Type getReturnType() {
        return Types.TEXT;
      }
    });
  }

  @Test
  public void canSelectInADifferentAttributeOrder() throws Exception {
    Struct sourceType = Struct.of("foo1", Types.TEXT)
        .and("foo2", Types.TEXT)
        .and("foo3", Types.TEXT)
        .and("foo4", Types.TEXT);

    Struct expectedReturnType = Struct.of("foo4", Types.TEXT)
        .and("foo2", Types.TEXT)
        .and("foo3", Types.TEXT)
        .and("foo1", Types.TEXT);

    ResultOrProblems<? extends Projector> projection = SelectProjector.realize(sourceType,
        toStructDeclaration("{foo4, foo2, foo3, foo1}"),
        expressionRealizer);

    assertThat(projection, result(Projector.class, projecting(expectedReturnType)));

    assertThat(projection.get().apply(Tuple.ofValues(sourceType, "one", "two", "three", "four")),
        is(Tuple.ofValues(expectedReturnType, "four", "two", "three", "one")));
  }

  @Test
  public void canSelectWithMathExpressions() throws Exception {
    Struct sourceType = Struct.of("x", Types.INTEGER)
        .and("y", Types.INTEGER);

    Struct expectedReturnType = Struct.of("x_plus_y", Types.INTEGER)
        .and("x_times_y", Types.INTEGER);

    ResultOrProblems<? extends Projector> projection = SelectProjector.realize(sourceType,
        toStructDeclaration("{x + y as x_plus_y, x * y as x_times_y}"),
        expressionRealizer);

    assertThat(projection, result(Projector.class, projecting(expectedReturnType)));

    assertThat(projection.get().apply(Tuple.ofValues(sourceType, 2L, 5L)),
        is(Tuple.ofValues(expectedReturnType, 7L, 10L)));
  }

  @Test
  public void canProjectAFunctionCall() throws Exception {
    Struct sourceType = Struct.of("foo1", Types.TEXT)
        .and("foo2", Types.TEXT)
        .and("foo3", Types.TEXT)
        .and("foo4", Types.TEXT);

    Struct expectedReturnType = Struct.of("dummy1", Types.TEXT)
        .and("dummy2", Types.TEXT);

    ResultOrProblems<? extends Projector> projection = SelectProjector.realize(sourceType,
        toStructDeclaration("{dummy(foo1, foo2) as dummy1, dummy(foo4, foo3) as dummy2}"),
        expressionRealizer);

    assertThat(projection, result(Projector.class, projecting(expectedReturnType)));

    assertThat(projection.get().apply(Tuple.ofValues(sourceType, "one", "two", "three", "four")),
        is(Tuple.ofValues(expectedReturnType, "one+two", "four+three")));
  }

  @Test
  public void failsIfOutputNamesAreReused() throws Exception {
    Struct sourceType = Struct.of("foo1", Types.TEXT)
        .and("foo2", Types.TEXT)
        .and("foo3", Types.TEXT)
        .and("foo4", Types.TEXT);

    StructDeclaration decl = toStructDeclaration("{foo4 as foo_bar, foo2 as foo_bar, foo3, foo1}");

    ResultOrProblems<? extends Projector> projection =
      SelectProjector.realize(sourceType, decl, expressionRealizer);

    assertThat(
      projection,
      failedResult(
        hasAncestorProblem(
          equalTo(ExpressionProblems.get().canNotReplaceMember("foo_bar", decl.getMember(0), decl.getMember(1)))
        )
      )
    );
  }

  StructDeclaration toStructDeclaration(String source) {
    return ExpressionParser.parseString(source).isA(StructDeclaration.class).get();
  }

}
