/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.projection;


import static nz.org.riskscape.engine.Matchers.*;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import org.junit.Test;
import org.locationtech.jts.geom.Point;

import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.gt.LatLongGeometryHelper;
import nz.org.riskscape.engine.gt.NZMGGeometryHelper;
import nz.org.riskscape.engine.types.Referenced;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.ResultOrProblems;

@SuppressWarnings("unchecked")
public class ForceSridProjectionTest extends ProjectTest {


  NZMGGeometryHelper nzmgHelper = new NZMGGeometryHelper(project.getSridSet());
  LatLongGeometryHelper latLongHelper = new LatLongGeometryHelper(project.getSridSet());

  ForceSridProjection subject = new ForceSridProjection(nzmgHelper.getCrs(), project.getSridSet());

  @Test
  public void canForceUnReferencedGeometryToNewSRID() {
    Struct type = Struct.of("geom", Types.POINT);

    Struct expected = Struct.of("geom", Referenced.of(Types.POINT, nzmgHelper.getCrs()));

    ResultOrProblems<Projector> projector = subject.getProjectionFunction(type);
    assertFalse(projector.hasProblems());
    assertThat(projector.get(), projecting(expected));

    Tuple projected = projector.get().apply(Tuple.ofValues(type, latLongHelper.point(10, 10)));
    Point projectedPoint = (Point)projected.fetch("geom");
    assertThat(projectedPoint.getX(), is(10D));
    assertThat(projectedPoint.getY(), is(10D));
    assertThat(projectedPoint.getSRID(), is(project.getSridSet().get(nzmgHelper.getCrs())));
  }

  @Test
  public void canForceReferencedGeometryToNewSRID() {
    Struct type = Struct.of("geom", Referenced.of(Types.POINT, latLongHelper.getCrs()));

    Struct expected = Struct.of("geom", Referenced.of(Types.POINT, nzmgHelper.getCrs()));

    ResultOrProblems<Projector> projector = subject.getProjectionFunction(type);
    assertFalse(projector.hasProblems());
    assertThat(projector.get(), projecting(expected));

    Tuple projected = projector.get().apply(Tuple.ofValues(type, latLongHelper.point(10, 10)));
    Point projectedPoint = (Point)projected.fetch("geom");
    assertThat(projectedPoint.getX(), is(10D));
    assertThat(projectedPoint.getY(), is(10D));
    assertThat(projectedPoint.getSRID(), is(project.getSridSet().get(nzmgHelper.getCrs())));
  }

  @Test
  public void canProjectTupleWithUnexpectedNulls() {
    Struct type = Struct.of("geom", Types.POINT, "value", Types.TEXT);

    Struct expected = Struct.of("geom", Referenced.of(Types.POINT, nzmgHelper.getCrs()),
        "value", Types.TEXT);

    ResultOrProblems<Projector> projector = subject.getProjectionFunction(type);
    assertFalse(projector.hasProblems());
    assertThat(projector.get(), projecting(expected));

    Tuple projected = projector.get().apply(Tuple.ofValues(type, latLongHelper.point(10, 10), "foo"));
    Point projectedPoint = (Point)projected.fetch("geom");
    assertThat(projectedPoint.getX(), is(10D));
    assertThat(projectedPoint.getY(), is(10D));
    assertThat(projectedPoint.getSRID(), is(project.getSridSet().get(nzmgHelper.getCrs())));
    assertThat(projected.fetch("value"), is("foo"));

    // value is null
    projected = projector.get().apply(Tuple.ofValues(type, latLongHelper.point(10, 10)));
    projectedPoint = (Point)projected.fetch("geom");
    assertThat(projectedPoint.getX(), is(10D));
    assertThat(projectedPoint.getY(), is(10D));
    assertThat(projectedPoint.getSRID(), is(project.getSridSet().get(nzmgHelper.getCrs())));
    assertThat(projected.fetch("value"), nullValue());

    // geom is null
    projected = projector.get().apply(Tuple.ofValues(type, null, "foo"));
    assertThat(projected.fetch("geom"), nullValue());
    assertThat(projected.fetch("value"), is("foo"));
  }


}
