/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.plugin;

import static org.junit.Assert.*;

import java.io.File;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.jar.Manifest;

import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;

import com.google.common.collect.Sets;

import nz.org.riskscape.engine.Assert;
import nz.org.riskscape.engine.Engine;
import nz.org.riskscape.problem.Problem;

@SuppressWarnings("unchecked")
public class PluginRepositoryTest {

  PluginRepository repository = new PluginRepository();
  URL pwd;

  ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
  PluginClassLoader pluginClassLoader = PluginClassLoader.newDummy();
  Manifest manifest = new Manifest();
  PluginDescriptor descriptor;
  Plugin built;

  @Before
  public void setup() throws Exception {
    pwd = new File(".").toURI().toURL();
    manifest.getMainAttributes().putValue("Riskscape-Plugin", EmptyPlugin.class.getName());
    descriptor = Mockito.mock(PluginDescriptor.class);
  }

  public static class EmptyPlugin extends Plugin {

    public EmptyPlugin(PluginDescriptor pDescriptor) {
      super(pDescriptor);
    }

    @Override
    public List<Problem> initializeEngine(Engine engine) {
      return Collections.emptyList();
    }

    @Override
    public void startUp(Map<String, List<String>> settings) throws PluginRuntimeException {
    }
  }

  Plugin plugin;

  @Test
  public void getActivedPluginReturnsPluginInstance() throws Exception {
    Mockito.when(descriptor.newPluginInstance()).thenAnswer(args -> new EmptyPlugin(descriptor));
    repository.activatePlugin(descriptor, p -> plugin = p);
    assertSame(plugin, repository.getActivatedPlugin(EmptyPlugin.class).get());
  }

  @Test
  public void getActivedPluginReturnsEmptyForFailedPlugin() throws Exception {
    Assert.assertThrows(PluginRuntimeException.class,
        () -> repository.activatePlugin(descriptor,
            p -> {
              throw new PluginRuntimeException("Failed");
            }));

    assertFalse(repository.getActivatedPlugin(EmptyPlugin.class).isPresent());
    assertNull(plugin);
  }

  @Test
  public void aPluginWithDependenciesIsSetupCorrectly() throws Exception {
    Manifest dependentManifest = new Manifest();
    dependentManifest.getMainAttributes().putValue("Riskscape-Plugin", EmptyPlugin.class.getName());
    dependentManifest.getMainAttributes().putValue("Riskscape-Plugin-Dependency-IDs", "bar-plugin");
    Manifest dependencyManifest = new Manifest();
    dependencyManifest.getMainAttributes().putValue("Riskscape-Plugin", EmptyPlugin.class.getName());
    dependencyManifest.getMainAttributes().putValue("Riskscape-Plugin-ID", "bar-plugin");

    PluginClassLoader dependentLoader = PluginClassLoader.newDummy();
    PluginClassLoader dependencyLoader = PluginClassLoader.newDummy();

    PluginDescriptor dependent = new DefaultPluginDescriptor(pwd, dependentLoader, dependentManifest);
    PluginDescriptor dependency = new DefaultPluginDescriptor(pwd, dependencyLoader, dependencyManifest);

    repository.registeredPlugins.add(dependent);
    repository.registeredPlugins.add(dependency);

    List<PluginDescriptor> plugins = new ArrayList<>();
    repository.activatePlugin(dependent, p-> plugins.add(p.getDescriptor()));

    // both should have been loaded
    assertEquals(Sets.newHashSet(dependent, dependency), Sets.newHashSet(plugins));

    // class loading gets 'connected up', too
    assertTrue(dependentLoader.getDependencies().contains(dependencyLoader));

    // check each plugin should only be loaded *once*
    repository.activatePlugin(dependency, p -> plugins.add(p.getDescriptor()));
    assertEquals(2, plugins.size());
  }

  @Test
  public void aPluginWithDependenciesIsOnlyLoadedOnce() throws Exception {
    Manifest dependentManifest = new Manifest();
    dependentManifest.getMainAttributes().putValue("Riskscape-Plugin", EmptyPlugin.class.getName());
    dependentManifest.getMainAttributes().putValue("Riskscape-Plugin-Dependency-IDs", "bar-plugin");
    Manifest dependencyManifest = new Manifest();
    dependencyManifest.getMainAttributes().putValue("Riskscape-Plugin", EmptyPlugin.class.getName());
    dependencyManifest.getMainAttributes().putValue("Riskscape-Plugin-ID", "bar-plugin");

    PluginClassLoader dependentLoader = PluginClassLoader.newDummy();
    PluginClassLoader dependencyLoader = PluginClassLoader.newDummy();

    PluginDescriptor dependent = new DefaultPluginDescriptor(pwd, dependentLoader, dependentManifest);
    PluginDescriptor dependency = new DefaultPluginDescriptor(pwd, dependencyLoader, dependencyManifest);

    repository.registeredPlugins.add(dependent);
    repository.registeredPlugins.add(dependency);

    // the above code is the same as aPluginWithDependenciesIsSetupCorrectly() but this time we load
    // the plugins in the other order - the dependency first, then the dependent
    List<PluginDescriptor> plugins = new ArrayList<>();
    repository.activatePlugin(dependency, p -> plugins.add(p.getDescriptor()));
    assertEquals(Arrays.asList(dependency), plugins);

    // check each plugin should only be loaded *once*
    repository.activatePlugin(dependent, p -> plugins.add(p.getDescriptor()));
    assertEquals(Arrays.asList(dependency, dependent), plugins);
  }

}
