/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.pipeline;

import static org.junit.Assert.*;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.function.Function;

import org.junit.Test;

import nz.org.riskscape.pipeline.PipelineParser;
import nz.org.riskscape.pipeline.ast.PipelineDeclaration;
import nz.org.riskscape.pipeline.ast.StepDefinition;

public class DefaultStepNamingPolicyTest {

  String pipeline;
  Set<String> taken = new HashSet<>();

  @Test
  public void assignsStepIdAsName() {
    pipeline = "foo()";
    assertStepNames("foo");

    pipeline = "foo() bar() baz()";
    assertStepNames("foo", "bar", "baz");
  }

  @Test
  public void multiplesOfTheSameIdAreEnumerated() {
    // in their own chains
    pipeline = "foo() foo() foo()";
    assertStepNames("foo", "foo_1", "foo_2");

    // in the same chain
    pipeline = "foo() -> foo() -> foo()";
    assertStepNames("foo", "foo_1", "foo_2");

    // references don't count
    pipeline = "foo() -> foo foo() -> foo foo() -> foo";
    assertStepNames("foo", "foo_1", "foo_2");

    // respects already taken names
    pipeline = "foo() bar()";
    taken.add("foo");
    assertStepNames("foo_1", "bar");
  }

  @Test
  public void considersNamedStepsInTheCounting() throws Exception {
    pipeline = "foo() -> foo() as bar -> foo() as baz -> foo() -> foo()";
    assertStepNames("foo", null, null, "foo_1", "foo_2");

    pipeline = "foo() as foo_2 -> foo() as foo -> foo() as foo_1 -> foo() -> foo() as foo_3";
    assertStepNames(null, null, null, "foo_4", null);
  }

  @Test
  public void inTheUnlikelyEventOfFunnyIdsItStillWorks() throws Exception {
    pipeline = "\"this Step Is Odd!\"()";
    assertStepNames("this Step Is Odd!");
  }

  private void assertStepNames(String... names) {
    PipelineDeclaration ast = PipelineParser.INSTANCE.parsePipeline(pipeline);
    Iterator<String> expectedNames = Arrays.asList(names).iterator();
    Iterator<StepDefinition> sdIterator = ast.stepDefinitionIterator();

    Function<StepDefinition, String> assigned = new DefaultStepNamingPolicy(taken).getStepNameFunction(ast);

    while (expectedNames.hasNext()) {
      String expected = expectedNames.next();

      StepDefinition defn = sdIterator.next();
      if (expected == null) {
        assertTrue("defn should have a name", defn.getNameToken().isPresent());
        continue;
      }

      assertEquals(
        defn.toString(),
        expected,
        assigned.apply(defn)
      );
    }

    assertFalse(sdIterator.hasNext());
  }

}
