/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.output;

import static nz.org.riskscape.engine.output.ShapeFileNullMapper.*;

import static org.hamcrest.Matchers.*;

import org.junit.Test;

import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

import org.locationtech.jts.geom.Point;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

import nz.org.riskscape.engine.types.Referenced;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.types.WithinRange;
import nz.org.riskscape.engine.types.WithinSet;

public class ShapeFileNullMapperTest {

  ShapeFileNullMapper mapper = new ShapeFileNullMapper();

  Point point = mock(Point.class);
  CoordinateReferenceSystem crs = mock(CoordinateReferenceSystem.class);

  @Test
  public void returnsNonNullValue() {
    assertThat(mapper.mapValueIfNecessary("foo", Types.TEXT), is("foo"));
    assertThat(mapper.mapValueIfNecessary(20L, Types.INTEGER), is(20L));

    assertThat(mapper.mapValueIfNecessary(point, Types.GEOMETRY), is(point));
    assertThat(mapper.mapValueIfNecessary(point, Types.POINT), is(point));
  }

  @Test
  public void returnsNullMappedTextValue() {
    assertThat(mapper.mapValueIfNecessary(null, Types.TEXT), is(""));
  }

  @Test
  public void returnsNullMappedBooleanValue() {
    assertThat(mapper.mapValueIfNecessary(null, Types.BOOLEAN), is(Boolean.FALSE));
  }

  @Test
  public void returnsNullMappedNumberValue() {
    assertThat(mapper.mapValueIfNecessary(null, Types.INTEGER), is(-9999L));
    assertThat(mapper.mapValueIfNecessary(null, Types.FLOATING), is(-9999L));
  }

  @Test
  public void returnsNullMappedValueForWrappedType() {
    assertThat(mapper.mapValueIfNecessary(null, new WithinRange(Types.INTEGER, 0L, 10L)), is(-9999L));
    assertThat(mapper.mapValueIfNecessary(null, new WithinSet(Types.INTEGER, 0L, 10L, 20L)), is(-9999L));
    assertThat(mapper.mapValueIfNecessary(null, new WithinSet(Types.TEXT, "foo", "bar")), is(""));
  }

  @Test
  public void returnsNullMappedGeometry() {
    assertThat(mapper.mapValueIfNecessary(null, Types.GEOMETRY), is(NULL_VALUE_GEOMETRY));
    assertThat(mapper.mapValueIfNecessary(null, Types.POINT), is(NULL_VALUE_GEOMETRY));
    assertThat(mapper.mapValueIfNecessary(null, Types.POLYGON), is(NULL_VALUE_GEOMETRY));

    assertThat(mapper.mapValueIfNecessary(null, Referenced.of(Types.POINT, crs)), is(NULL_VALUE_GEOMETRY));
  }

}
