/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.output;

import static org.hamcrest.Matchers.contains;
import static org.junit.Assert.*;
import static org.mockito.ArgumentMatchers.*;
import static org.mockito.ArgumentMatchers.any;

import java.net.URI;

import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;
import org.mockito.stubbing.OngoingStubbing;

import nz.org.riskscape.engine.OutputProblems;
import nz.org.riskscape.engine.pipeline.RealizedPipeline;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;

public class DefaultPipelineOutputStoresTest {

  DefaultPipelineOutputStores stores = new DefaultPipelineOutputStores();
  RealizedPipeline mockPipeline = Mockito.mock(RealizedPipeline.class);
  PipelineOutputOptions options = new PipelineOutputOptions("test");
  PipelineOutputStore mockFooStore = Mockito.mock(PipelineOutputStore.class, "foo store");
  PipelineOutputStore mockBarStore = Mockito.mock(PipelineOutputStore.class, "bar store");
  PipelineOutputContainer mockContainer = Mockito.mock(PipelineOutputContainer.class);

  @Before
  public void setup() {
    Mockito.when(mockFooStore.getId()).thenReturn("foo");
    Mockito.when(mockFooStore.isApplicable(any())).thenReturn(PipelineOutputStore.PRIORITY_NA);
    stores.add(mockFooStore);

    Mockito.when(mockBarStore.getId()).thenReturn("bar");
    Mockito.when(mockBarStore.isApplicable(any())).thenReturn(PipelineOutputStore.PRIORITY_NA);
    stores.add(mockBarStore);
  }


  @Test
  public void createReturnsAContainerFromTheHighestPriorityStore() throws Exception {
    whenCreateReturnOptions(URI.create("foo:bar"), mockFooStore, PipelineOutputStore.PRIORITY_DEFAULT);
    whenCreateReturnOptions(URI.create("foo:bar"), mockBarStore, PipelineOutputStore.PRIORITY_HIGH);

    assertSame(mockContainer, stores.create(URI.create("foo:bar"), mockPipeline, store -> {
      // bar should have won
      assertSame(store, mockBarStore);
      return options;
    }).get());
  }

  @Test
  public void createReturnsAContainerFromTheHighestPriorityStore_ReverseOrder() throws Exception {
    // do the previous test, but this time flip the priorities to make sure it wasn't an accident
    whenCreateReturnOptions(URI.create("foo:bar"), mockBarStore, PipelineOutputStore.PRIORITY_DEFAULT);
    whenCreateReturnOptions(URI.create("foo:bar"), mockFooStore, PipelineOutputStore.PRIORITY_HIGH);

    assertSame(mockContainer, stores.create(URI.create("foo:bar"), mockPipeline, store -> {
      assertSame(store, mockFooStore);
      return options;
    }).get());
  }


  @Test
  public void createFailsIfNoStoreSupportsTheURI() throws Exception {
    assertThat(
      stores.create(URI.create("foo:bar"), mockPipeline, store -> {
        fail("nothing is supposed to support this");
        return null;
      }).getProblems(),
      contains(OutputProblems.get().outputLocationNotSupported(URI.create("foo:bar")))
    );
  }

  @Test
  public void createReturnsAnyErrorsFromTheStore() throws Exception {
    whenCreate(URI.create("foo:bar"), mockFooStore, PipelineOutputStore.PRIORITY_DEFAULT)
      .thenReturn(ResultOrProblems.failed(Problems.foundWith("this and that")));

    assertThat(
      stores.create(URI.create("foo:bar"), mockPipeline, store -> {
        return options;
      }).getProblems(),
      contains(Problems.foundWith("this and that"))
    );
  }

  private OngoingStubbing<ResultOrProblems<PipelineOutputContainer>> whenCreate(
      URI create,
      PipelineOutputStore mockStore,
      int priority
  ) {
    Mockito.when(mockStore.isApplicable(create)).thenReturn(priority);
    return Mockito.when(mockStore.create(eq(create), same(mockPipeline), same(options)));
  }


  private void whenCreateReturnOptions(URI create, PipelineOutputStore mockStore, int priority) {
    whenCreate(create, mockStore, priority).thenReturn(ResultOrProblems.of(mockContainer));
  }
}
