/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.output;

import static nz.org.riskscape.engine.output.CsvWriter.*;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.Point;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

import nz.org.riskscape.engine.CrsHelper;
import nz.org.riskscape.engine.SRIDSet;
import nz.org.riskscape.engine.TemporaryDirectoryTestHelper;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.resource.FileCreateHandle;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.types.WithinRange;
import nz.org.riskscape.engine.types.WithinSet;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ProblemSink;
import nz.org.riskscape.problem.Problem.Severity;

public class CsvWriterTest implements CrsHelper,TemporaryDirectoryTestHelper  {

  Path tmpDir;
  File tmpfile;
  private CoordinateReferenceSystem crs;
  private Point point;
  private GeometryFactory geometryFactory;
  SRIDSet sridSet = new SRIDSet();
  List<Problem> collectedProblems = new ArrayList<>();
  ProblemSink sink = p -> collectedProblems.add(p);


  @Before
  public void setup() throws IOException {
    tmpDir = createTempDirectory("CsvWriterTest");
    tmpfile = File.createTempFile(getClass().getSimpleName(), ".csv", tmpDir.toFile());
    crs = nzMapGrid();
    geometryFactory = sridSet.getGeometryFactory(crs);
    point = geometryFactory.createPoint(new Coordinate(0, 0));
  }

  @After
  public void teardown() throws Exception {
    remove(tmpDir);
  }

  @Test
  public void canWriteOutATableOfIntegersAndPoints() throws IOException {

    Struct struct = Struct.of("value", Types.INTEGER).and("geometry", Types.GEOMETRY).build();

    List<Integer> values = Arrays.asList(1, 2, 3, 4, 5, 6);
    try (CsvWriter writer = createCsvWriter(struct)) {
      values.forEach(integer -> writer.write(new Tuple(struct).set("value", integer).set("geometry", point)));
    }

    List<String> lines = Files.readAllLines(tmpfile.toPath());

    assertEquals(Arrays.asList(
        "value,geometry",
        "1,POINT (0 0)",
        "2,POINT (0 0)",
        "3,POINT (0 0)",
        "4,POINT (0 0)",
        "5,POINT (0 0)",
        "6,POINT (0 0)"),
        lines);
  }

  @Test
  public void canWriteOutATableOfStructs() throws IOException {

    WithinSet simpleSet = new WithinSet(Types.TEXT, "foo", "bar", "baz");
    WithinRange ranged = new WithinRange(Types.INTEGER, 0, 100);

    Struct myStruct = Struct.of("int", Types.INTEGER)
        .and("string", Types.TEXT)
        .and("set", simpleSet)
        .and("ranged", ranged)
        .and("the_geom", Types.GEOMETRY)
        .build();

    List<Tuple> values = Arrays.asList(
        new Tuple(myStruct).set("int", 3).set("string", "cool").set("set", "bar").set("ranged", 66)
          .set("the_geom", point),
        new Tuple(myStruct).set("int", 63).set("string", "story").set("set", "baz").set("ranged", 33)
          .set("the_geom", point)
    );

    try (CsvWriter writer = createCsvWriter(myStruct)) {
      values.forEach(value-> writer.write(value));
    }


    List<String> lines = Files.readAllLines(tmpfile.toPath());

    assertEquals(Arrays.asList(
        "int,string,set,ranged,the_geom",
        "3,cool,bar,66,POINT (0 0)",
        "63,story,baz,33,POINT (0 0)"),
        lines);
  }

  @Test
  public void willQuoteStringsThatIncludeSpecialCharacters() throws Exception {
    Struct struct = Types.TEXT.asStruct();

    try (CsvWriter writer = createCsvWriter(struct)) {
      writer.write(Tuple.ofValues(struct, ","));
      writer.write(Tuple.ofValues(struct, "\""));
      writer.write(Tuple.ofValues(struct, "foo\nbar"));
    }

    List<String> lines = Files.readAllLines(tmpfile.toPath());

    assertEquals(Arrays.asList(
        "value",
        "\",\"",
        "\"\"\"\"",
        "\"foo",
        "bar\""),
        lines);
  }

  @Test
  public void willPrintOutNestedStructsByDotSeparatingThePathInTheHeader() throws Exception {
    Struct first = Types.TEXT.asStruct();
    Struct second = Types.TEXT.asStruct();

    Struct struct = first.parent("first").add("second", second);

    try (CsvWriter writer = createCsvWriter(struct)) {
      writer.write(Tuple.ofValues(struct, Tuple.ofValues(first, "foo"), Tuple.ofValues(second, "bar")));
    }

    List<String> lines = Files.readAllLines(tmpfile.toPath());

    assertEquals(Arrays.asList(
        "first.value,second.value",
        "foo,bar"),
        lines);
  }

  @Test
  public void willCorrectlyHandleNullMembersInNestedStricts() throws Exception {
    Struct first = Types.TEXT.asStruct();
    Struct second = Types.TEXT.asStruct();

    Struct struct = first.parent("first").add("second", second);

    try (CsvWriter writer = createCsvWriter(struct)) {
      writer.write(Tuple.ofValues(struct, Tuple.ofValues(first, "foo"), Tuple.ofValues(second, "bar")));
      writer.write(Tuple.ofValues(struct));
      writer.write(Tuple.ofValues(struct, Tuple.ofValues(first, "foo")));
      writer.write(Tuple.ofValues(struct, null, Tuple.ofValues(first, "bar")));
    }

    List<String> lines = Files.readAllLines(tmpfile.toPath());

    assertEquals(Arrays.asList(
        "first.value,second.value",
        "foo,bar",
        ",",
        "foo,",
        ",bar"
        ),
        lines);
  }


  @Test
  public void skipsEmptyValues() throws IOException {
    Struct struct = Struct.of("foo", Types.TEXT).and("bar", Types.TEXT).build();

    try (CsvWriter writer = createCsvWriter(struct)) {
      writer.write(Tuple.of(struct));
      writer.write(Tuple.ofValues(struct, "cool", null));
      writer.write(Tuple.ofValues(struct, null, "story"));

    }

    List<String> lines = Files.readAllLines(tmpfile.toPath());
    assertEquals(Arrays.asList(
        "foo,bar",
        ",",
        "cool,",
        ",story"),
        lines);
  }

  @Test
  public void csvWithNoFeaturesHasHeader() throws IOException {
    Struct struct = Struct.of("foo", Types.TEXT).and("bar", Types.TEXT).build();

    CsvWriter writer = createCsvWriter(struct);
    writer.close();

    List<String> lines = Files.readAllLines(tmpfile.toPath());
    assertEquals(Arrays.asList("foo,bar"), lines);
  }

  @Test
  public void willTruncateVeryLongValues() throws IOException {
    Struct struct = Struct.of("foo", Types.TEXT);

    String longString = IntStream.range(0, 1500)
        .mapToObj(i -> "The cat jumped over the dog")
        .collect(Collectors.joining(","));

    // Sanity check that the test input is larger than allowed
    assertThat(longString.length(), greaterThan(COLUMN_MAX_CHARACTERS));

    try (CsvWriter writer = createCsvWriter(struct)) {
      writer.write(Tuple.ofValues(struct, longString));
      writer.write(Tuple.ofValues(struct, longString));
      writer.write(Tuple.ofValues(struct, longString));
    }

    List<String> lines = Files.readAllLines(tmpfile.toPath());
    assertThat(lines, hasItems(
        is("foo"),
        hasLength(COLUMN_MAX_CHARACTERS + 1)    // + 1 to take account of the line feed that is not
                                                // part of the column
    ));

    assertThat(collectedProblems, contains(
      CsvWriter.PROBLEMS.dataTruncated("foo", 1).withSeverity(Severity.WARNING),
      CsvWriter.PROBLEMS.dataTruncated("foo", 2).withSeverity(Severity.WARNING),
      CsvWriter.PROBLEMS.dataTruncated("foo", 3).withSeverity(Severity.WARNING)
    ));
  }

  @Test
  public void willStopProducingTruncationWarningsAfter100() throws IOException {
    Struct struct = Struct.of("foo", Types.TEXT);

    String longString = IntStream.range(0, 1500)
        .mapToObj(i -> "The cat jumped over the dog")
        .collect(Collectors.joining(","));

    // Sanity check that the test input is larger than allowed
    assertThat(longString.length(), greaterThan(COLUMN_MAX_CHARACTERS));

    try (CsvWriter writer = createCsvWriter(struct)) {
      for (int i = 0; i < 500; i++) {
        writer.write(Tuple.ofValues(struct, longString));
        writer.write(Tuple.ofValues(struct, longString));
        writer.write(Tuple.ofValues(struct, longString));
      }
    }

    List<Problem> expectedProblems = new ArrayList<>(101);
    IntStream.rangeClosed(1, 100)
        .mapToObj(idx -> CsvWriter.PROBLEMS.dataTruncated("foo", idx).withSeverity(Severity.WARNING))
        .forEach(p -> expectedProblems.add(p));

    expectedProblems.add(CsvWriter.PROBLEMS.warningLimitReached(100).withSeverity(Severity.WARNING));
    assertThat(collectedProblems, equalTo(expectedProblems));
  }

  @Test
  public void canWriteToGivenWriter() throws Exception {
    StringWriter outputWriter = new StringWriter();

    Struct struct = Struct.of("foo", Types.TEXT).and("bar", Types.TEXT).build();

    try (CsvWriter writer = new CsvWriter(struct, sink, outputWriter)) {
      writer.write(Tuple.ofValues(struct, "cool", "story"));
    }

    assertThat(
      outputWriter.toString(), is(
          CsvWriter.BOM +
          """
          foo,bar
          cool,story
          """
      )
    );
  }

  private CsvWriter createCsvWriter(Struct type) {
    return new CsvWriter(type, sink, new FileCreateHandle(null, tmpfile), false);
  }
}
