/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.output;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import org.geotools.referencing.CRS;
import org.junit.Test;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;

import nz.org.riskscape.engine.CrsHelper;
import nz.org.riskscape.engine.SRIDSet;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ProblemSink;

public class AxisSwapperTest implements CrsHelper {

  GeometryFactory gf = new GeometryFactory();
  // crs is just metadata, safe for it to be garbage
  AxisSwapper subject = new AxisSwapper(SRIDSet.EPSG2193_NZTM);

  Coordinate c1 = new Coordinate(100, 50);
  Coordinate c2 = new Coordinate(100, 150);
  Coordinate c3 = new Coordinate(200, 50);

  URI storedAt = URI.create("test");
  List<Problem> problems = new ArrayList<>();
  ProblemSink problemSink = p -> problems.add(p);

  @Test
  public void getIfNeededReturnsEmptyIfAlreadyXYOrder() throws Exception {
    assertFalse(AxisSwapper.getForceXY(SRIDSet.EPSG4326_LONLAT, storedAt, problemSink).isPresent());
  }

  @Test
  public void getIfNeededReturnsSwapperIfInYXOrder() throws Exception {
    AxisSwapper swapper = AxisSwapper.getForceXY(SRIDSet.EPSG4326_LATLON, storedAt, problemSink).get();
    assertNotNull(swapper);
    assertTrue(CRS.equalsIgnoreMetadata(SRIDSet.EPSG4326_LONLAT, swapper.getNewCrs()));
  }

  @Test
  public void getIfNeededReturnsSwapperIfInYXOrderAndUnknownCRS() throws Exception {
    AxisSwapper swapper = AxisSwapper.getForceXY(customCrs(), storedAt, problemSink).get();
    assertNotNull(swapper);
    assertEquals(customCrs(), swapper.getNewCrs());
  }

  @Test
  public void canSwapPointAxis() {
    Point p = swap(gf.createPoint(c1));
    assertThat(p.toString(), is("POINT (50 100)"));
  }

  @Test
  public void canSwapLineAxis() {
    LineString line = swap(gf.createLineString(new Coordinate[] {c1, c2}));
    assertThat(line.toString(), is("LINESTRING (50 100, 150 100)"));
  }

  @Test
  public void canSwapPolygonAxis() {
    Polygon poly = swap(gf.createPolygon(new Coordinate[] {c1, c2, c3, c1}));
    assertThat(poly.toString(), is("POLYGON ((50 100, 150 100, 50 200, 50 100))"));
  }

  @Test
  public void canGetSwapperToSwapToCrsDefinedOrder() {
    // You don't need a swapper if the crs has the EPSG defined axis order
    assertFalse(AxisSwapper.getForceEpsgDefined(SRIDSet.EPSG4326_LATLON).isPresent());
    assertFalse(AxisSwapper.getForceEpsgDefined(SRIDSet.EPSG2193_NZTM).isPresent());

    // we will test by going around the block EPSG:4326 -> CRS:84 -> EPSG:4326
    AxisSwapper toCrs84 = AxisSwapper.getForceXY(SRIDSet.EPSG4326_LATLON, null, ProblemSink.DEVNULL).get();
    AxisSwapper toEpsg4326 = AxisSwapper.getForceEpsgDefined(toCrs84.getNewCrs()).get();

    // first we swap the point with the xy swapper
    Point p = swap(gf.createPoint(c1), toCrs84);
    assertThat(p.toString(), is("POINT (50 100)"));

    // and now we swap it back to EPSG:4326 defined order
    p = swap(p, toEpsg4326);
    assertThat(p.toString(), is("POINT (100 50)"));
  }

  private <T extends Geometry> T swap(T geom) {
    return swap(geom, subject);
  }

  private <T extends Geometry> T swap(T geom, AxisSwapper swapper) {
    return (T)swapper.swapAxis(geom);
  }

}
