/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.join;

import static nz.org.riskscape.engine.Matchers.*;
import static nz.org.riskscape.engine.types.Struct.*;

import static org.junit.Assert.*;

import java.util.Arrays;

import org.junit.Test;
import org.mockito.Mockito;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.problem.Problem.Severity;

@SuppressWarnings("unchecked")
public class JoinerTest {

  Struct lhsType = Struct.of("foo", Types.TEXT);
  Struct rhsType = Struct.of("bar", Types.TEXT);

  Joiner join = new Joiner(
      lhsType,
      rhsType,
      JoinCondition.CARTESIAN);

  @Test
  public void canModelAnInnerJoin() {
    assertEquals(of("foo", Types.TEXT, "bar", Types.TEXT), join.getJoinedType());
  }

  @Test
  public void canModelALeftOuterJoin() {
    join = join.withJoinType(JoinType.LEFT_OUTER);
    assertEquals(of("foo", Types.TEXT, "bar", Nullable.TEXT), join.getJoinedType());
  }

  @Test
  public void canJoinTwoTuples() throws Exception {
    Tuple lhs = Tuple.ofValues(lhsType, "foos");
    Tuple rhs = Tuple.ofValues(rhsType, "bars");
    Tuple joined = join.join(lhs, rhs);

    assertEquals("foos", joined.fetch("foo"));
    assertEquals("bars", joined.fetch("bar"));
    assertEquals(2, joined.size());
  }

  @Test
  public void canLeftOuterJoinTwoTuples() throws Exception {
    join = join.withJoinType(JoinType.LEFT_OUTER);
    Tuple lhs = Tuple.ofValues(lhsType, "foos");
    Tuple rhs = Tuple.ofValues(rhsType, "bars");
    Tuple joined = join.join(lhs, rhs);

    assertEquals("foos", joined.fetch("foo"));
    assertEquals("bars", joined.fetch("bar"));
    assertEquals(2, joined.size());
  }

  @Test
  public void canLeftOuterJoinTwoTuplesWithNullRhs() throws Exception {
    join = join.withJoinType(JoinType.LEFT_OUTER);
    Tuple lhs = Tuple.ofValues(lhsType, "foos");
    Tuple rhs = null;
    Tuple joined = join.join(lhs, rhs);

    assertEquals("foos", joined.fetch("foo"));
    assertNull(joined.fetch("bar"));
    assertEquals(2, joined.size());
  }

  @Test
  public void canBuildWithResultOr() throws Exception {
    ResultOrProblems<Joiner> resultOr = Joiner.build(lhsType, rhsType, JoinCondition.CARTESIAN, JoinType.INNER);
    join = resultOr.get();
    assertEquals(of("foo", Types.TEXT, "bar", Types.TEXT), join.getJoinedType());
  }

  @Test
  public void buildCanFailIfJoinedTypeEndsUpWithDuplicates() throws Exception {
    assertThat(Joiner.build(lhsType, lhsType, JoinCondition.CARTESIAN, JoinType.INNER),
        failedResult(isProblem(Severity.ERROR, ProblemCodes.DUPLICATE_KEYS)));
  }


  @Test
  public void buildCanFailIfConditionIsInvalid() throws Exception {
    JoinCondition joinCondition = Mockito.mock(JoinCondition.class);
    Mockito.when(joinCondition.validate(Mockito.any())).thenReturn(Arrays.asList(Problem.error("Bad Times")));
    assertThat(Joiner.build(lhsType, rhsType, joinCondition, JoinType.INNER),
        failedResult(isProblem(Severity.ERROR, ProblemCodes.JOIN_CONDITION_INVALID)));

  }

}
