/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.join;

import static nz.org.riskscape.engine.Matchers.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Arrays;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.rl.DefaultOperators;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.problem.Problem.Severity;
import nz.org.riskscape.rl.ast.ExpressionProblems;

@SuppressWarnings("unchecked")
public class ExpressionJoinConditionTest extends ProjectTest {

  @Before
  public void setup() {
    project.getFunctionSet().insertFirst(DefaultOperators.INSTANCE);
  }

  @Test
  public void goodJoinCondition() {
    Struct inputType = Struct.of("foo", Types.TEXT, "bar", Types.TEXT);

    ResultOrProblems<ExpressionJoinCondition> join = realizeJoin("foo = bar", inputType);
    assertThat(join.hasProblems(), is(false));
    assertThat(join.get().validate(inputType), empty());

    assertThat(join.get().matches(Tuple.ofValues(inputType, "a", "a")), is(true));
    assertThat(join.get().matches(Tuple.ofValues(inputType, "a", "b")), is(false));
  }

  @Test
  public void validatingWithDifferentStructFails() {
    Struct inputType = Struct.of("foo", Types.TEXT, "bar", Types.TEXT);

    ResultOrProblems<ExpressionJoinCondition> join = realizeJoin("foo = bar", inputType);
    assertThat(join.hasProblems(), is(false));

    Struct inputTypeClone = Struct.of("foo", Types.TEXT, "bar", Types.TEXT);
    assertThat(join.get().validate(inputTypeClone), contains(
        isProblem(Severity.ERROR, is("Joined type not the same instance as realized with"))
    ));
  }

  @Test
  public void badJoinConditionCannotBeRealized_UnknownAttributes() {
    Struct inputType = Struct.of("foo", Types.TEXT, "bar", Types.TEXT);

    assertThat(realizeJoin("foo = barbar", inputType), failedResult(
        equalTo(ExpressionProblems.get().noSuchStructMember("barbar", Arrays.asList("foo", "bar")))
    ));
  }

  @Test
  public void badJoinConditionCannotBeRealized_NotBooleanResult() {
    Struct inputType = Struct.of("foo", Types.TEXT, "bar", Types.TEXT);

    assertThat(realizeJoin("foo", inputType), failedResult(
        isProblem(Severity.ERROR, is("Expression 'foo' must return Boolean. Found: Text"))));
  }

  ResultOrProblems<ExpressionJoinCondition> realizeJoin(String condition, Struct inputType) {
    return ExpressionJoinCondition.realize(expressionParser.parse(condition), inputType, expressionRealizer);
  }

}
