/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.io;

import static org.junit.Assert.*;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.HashMap;
import java.util.Map;

import org.junit.Test;

import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;

public class TupleSerializationTest {

  ByteArrayOutputStream bos = new ByteArrayOutputStream();
  @Test

  public void testBasicInAndOut() throws IOException {

    Struct testStruct = Struct
        .of("boolean", Types.BOOLEAN)
        .and("text", Types.TEXT)
        .and("decimal", Types.DECIMAL)
        .and("floating", Types.FLOATING)
        .and("geometry", Types.GEOMETRY)
        .and("integer", Types.INTEGER)
        .and("long", Types.INTEGER)
        .build();

    GeometryFactory factory = new GeometryFactory();
    org.locationtech.jts.geom.Point point = factory.createPoint(new Coordinate(55, 88));
    point.setSRID(55);

    Tuple subject = new Tuple(testStruct);
    subject.set("boolean", Boolean.TRUE).set("text", "hi mom").set("decimal", new BigDecimal("155.24242341"))
    .set("floating", 4D).set("geometry", point).set("integer", 4L).set("long", 23422526262626L);

    DataOutputStreamTupleWriter writer = newWriter();

    writer.write(subject);

    DataInputStreamTupleReader reader = newReader();

    Tuple readBackIn = reader.read(testStruct);

    assertEquals(subject, readBackIn);
    assertEquals(55, ((Geometry) readBackIn.fetch("geometry")).getSRID());
  }

  private DataOutputStreamTupleWriter newWriter() {
    return new DataOutputStreamTupleWriter(new DataOutputStream(bos));
  }

  private DataInputStreamTupleReader newReader() {
    return new DataInputStreamTupleReader(
        new DataInputStream(new ByteArrayInputStream(bos.toByteArray())));
  }

  @Test
  public void testNullablesInAndOut() throws IOException {
    Struct testStruct = Struct
        .of("boolean", Nullable.BOOLEAN)
        .and("decimal", Nullable.DECIMAL)
        .build();

    Tuple subjectWithNulls = new Tuple(testStruct);

    DataOutputStreamTupleWriter writer = newWriter();

    writer.write(subjectWithNulls);

    DataInputStreamTupleReader reader = newReader();

    Tuple readBackIn = reader.read(testStruct);

    assertEquals(subjectWithNulls, readBackIn);

    bos.reset();

    Tuple subjectWithoutNulls = new Tuple(testStruct)
        .set("boolean", Boolean.TRUE)
        .set("decimal", new BigDecimal("2.55"));

    writer = newWriter();
    writer.write(subjectWithoutNulls);
    reader = newReader();
    readBackIn = reader.read(testStruct);
    assertEquals(subjectWithoutNulls, readBackIn);
  }

  @Test
  public void numberTypesAreCoercedAtPointOfBeingWritten() throws IOException {
    Struct withNumberTypes = Struct.of("integer", Types.INTEGER).and("floating", Types.FLOATING).build();
    Tuple subject = Tuple.ofValues(withNumberTypes, 45, 45F);

    DataOutputStreamTupleWriter writer = newWriter();
    writer.write(subject);
    DataInputStreamTupleReader reader = newReader();
    Tuple readBackIn = reader.read(withNumberTypes);

    // it is read back in 'purely', so not entirely equivalent
    assertNotEquals(subject, readBackIn);
    Tuple expected = Tuple.ofValues(withNumberTypes, 45L, 45D);
    assertEquals(expected, readBackIn);
  }


  @Test
  public void testAnythingSerializableCanBeSerialized() throws IOException {
    Map<String, String> values = new HashMap<>();
    values.put("foo", "bar");

    Struct testStruct = Struct
        .of("anything", Types.ANYTHING)
        .build();

    Tuple subject = new Tuple(testStruct).set("anything", values);

    DataOutputStreamTupleWriter writer = newWriter();
    writer.write(subject);

    DataInputStreamTupleReader reader = newReader();

    Tuple readBackIn = reader.read(testStruct);

    assertEquals(subject, readBackIn);

  }

}
