/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.io;

import static org.junit.Assert.*;
import static nz.org.riskscape.engine.Assert.*;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Path;

import org.junit.Test;

import nz.org.riskscape.engine.TemporaryDirectoryTestHelper;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import org.junit.After;
import org.junit.Before;

public class DiskStorageTest implements TemporaryDirectoryTestHelper {

  Struct struct = Types.TEXT.asStruct();
  Tuple tuple = Tuple.ofValues(struct, "foo");

  Path tempDir;
  DiskStorage storage;

  @Before
  public void setup() throws IOException {
    tempDir = createTempDirectory("DiskStorageTest");
    storage = new DiskStorage(tempDir);
  }

  @After
  public void cleanUp() throws Exception {
    remove(tempDir);
  }

  @Test
  public void canCreateATemporaryWriter() throws IOException {
    DiskStorageWriter writer = (DiskStorageWriter) storage.createTemporaryWriter();
    writer.write(tuple);
    writer.close();

    assertTrue(writer.getFile().exists());
    Tuple read = storage.getReader(writer).read(struct);
    assertEquals("foo", read.fetch("value"));
  }

  @Test
  public void canOptionallyCompressOutput() throws IOException {
    storage = new DiskStorage(tempDir, true);
    DiskStorageWriter writer = (DiskStorageWriter) storage.createTemporaryWriter();
    writer.write(tuple);
    writer.close();

    assertTrue(writer.getFile().exists());
    Tuple read = storage.getReader(writer).read(struct);
    assertEquals("foo", read.fetch("value"));
  }

  @Test
  public void raisesASpecificErrorIfAWriterHasHadItsFileRemoved() throws IOException {
    DiskStorageWriter writer = (DiskStorageWriter) storage.createTemporaryWriter();
    writer.write(tuple);
    writer.close();

    assertTrue(writer.getFile().exists());

    writer.getFile().delete();

    FileNotFoundException error = assertThrows(FileNotFoundException.class, () -> storage.getReader(writer));
    assertEquals(
        "Attempting to create a reader from a writer with no file - has it been cleaned up?",
        error.getMessage());
  }


}
