/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.ini;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import org.ini4j.Ini;
import org.junit.Test;

import nz.org.riskscape.engine.resource.Resource;
import nz.org.riskscape.engine.resource.StringResource;

// kill this - replaced by ini4jconfig
public class IniParserTest {

  @Test
  public void parsingAllowSectionToHaveDuplicateKeys() throws Exception {
    Resource content = new StringResource("bookmark.ini",
        "# just a comment line",
        "[building]",
        "location=    file1.shp",
        "location                =    file2.shp            ");

    Ini ini = IniParser.parse(content.getContentStream());
    Ini.Section section = ini.get("building");
    //Not that you need to use getAll() to get each value
    List<String> values = section.getAll("location").stream().collect(Collectors.toList());
    assertThat(values, contains(
        "file1.shp",
        "file2.shp"));
  }

  @Test
  public void parsingAllowsDuplicationSections() throws Exception {
    Resource content = new StringResource("bookmark.ini",
        "# just a comment line",
        "[building]",
        "location=    file1.shp",
        "[building]",
        "location                =    file2.shp            ");

    Ini ini = IniParser.parse(content.getContentStream());
    //Not that you need to use getAll() to get each value
    List<String> values = new ArrayList<>();
    for (Ini.Section section : ini.getAll("building")) {
      values.add(section.get("location"));
    }
    assertThat(values, contains(
        "file1.shp",
        "file2.shp"));
  }

  @Test
  public void canGetAllSectionsFromIniEvenIfNamesAreDuplicated() throws Exception {
    Resource content = new StringResource("bookmark.ini",
        "# just a comment line",
        "[building]",
        "location=    file1.shp",
        "[building]",
        "location                =    file2.shp            ",
        "[type]",
        "location = type.ini");

    Ini ini = IniParser.parse(content.getContentStream());
    //Not that you need to use getAll() to get each value
    List<String> values = new ArrayList<>();
    for (Ini.Section section : ini.values()) {
      values.add(section.getName() + ":" + section.get("location"));
    }
    assertThat(values, contains(
        "building:file1.shp",
        "building:file2.shp",
        "type:type.ini"));
  }

  @Test
  public void parsingTrimsLeadAndTraillingWhiteSpace() throws Exception {
    Resource content = new StringResource("bookmark.ini",
        "# just a comment line",
        "[building]",
        "location1=    file1.shp",
        "           location2                =    file2.shp            ");

    Ini ini = IniParser.parse(content.getContentStream());
    Ini.Section section = ini.get("building");
    assertThat(section.get("location1"), is("file1.shp"));
    assertThat(section.get("location2"), is("file2.shp"));
  }

  @Test
  public void parsesMultilineValues() throws Exception {
    Resource content = new StringResource("bookmark.ini",
        "# just a comment line",
        "[building]",
        "    description = test description \\",
        "                  spanning two lines");

    Ini ini = IniParser.parse(content.getContentStream());
    Ini.Section section = ini.get("building");
    assertThat(section.get("description"), is("test description spanning two lines"));
  }

  @Test
  public void ensureThatWindowsPathDonotNeedEscaping() throws Exception {
    Resource iniResource = new StringResource("bookmark.ini",
        "# just a comment line",
        "[building]",
        "description = test description that continues \\",
        "              on the next line",
        "location=C:\\temp\\file.shp");

    Ini ini = IniParser.parse(iniResource.getContentStream());
    Ini.Section bm = ini.get("building");
    assertThat(bm.get("description"), is("test description that continues on the next line"));
    assertThat(bm.get("location"), is("C:\\temp\\file.shp"));
  }

}
