/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.i18n;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.net.MalformedURLException;
import java.nio.file.Paths;
import java.util.Locale;

import org.junit.Test;

public class RiskscapeMessageSourceTest {

  public static final Locale NZ = new Locale("en", "NZ");
  public static final Locale NZ_V1 = new Locale("en", "NZ", "variant1");
  public static final Locale JP = new Locale("ja", "JP");

  enum TestEnum {
    VALUE1,
    VALUE2,
    VALUE3
  }

  static class TestParameters {
    public TestEnum value;
  }

  I18nClassLoader classLoader = new I18nClassLoader();
  RiskscapeMessageSource subject = new RiskscapeMessageSource("messages", classLoader);;

  private ResourceClassLoader newResourceClassLoader(String pathSegment) throws MalformedURLException {
    return new ResourceClassLoader(
        "nz/org/riskscape/engine/i18n/" + pathSegment + "/",
        Paths.get("src", "test", "resources").toUri().toURL()
    );
  }

  private void init(String pathSegment) throws MalformedURLException {
    classLoader.append(pathSegment, newResourceClassLoader(pathSegment));
  }

  @Test
  public void returnsDefaultWhenCodeNotFound() throws MalformedURLException {
    init("one");

    assertThat(subject.getMessage("unknown", null, JP), is(nullValue()));
    assertThat(subject.getMessage("unknown", "place-holder var-arg"), is(nullValue()));
    assertThat(subject.getMessage("unknown"), is(nullValue()));

    assertThat(subject.getMessage("unknown", new Object[]{}, "default", JP), is("default"));
    assertThat(subject.getMessage("unknown", new Object[]{}, null, JP), is(nullValue()));

    assertThat(subject.getMessage(RiskscapeMessage.of("unknown")), is(nullValue()));
    assertThat(subject.getMessage(RiskscapeMessage.withDefault("unknown", "default")), is("default"));

    assertThat(
        subject.getMessage(RiskscapeMessage.withDefault("unknown1", "default")),
        is("default"));
  }

  @Test
  public void appliesArgumentReplacements() throws MalformedURLException {
    init("one");
    assertThat(subject.getMessage("with.args", new Object[] {"my-arg"}, JP), is("base bundle one my-arg"));
    assertThat(subject.getMessage("with.args", new Object[] {"MY-ARG"}, JP), is("base bundle one MY-ARG"));
  }

  @Test
  public void appliesArgumentReplacements_AndResolvesMessageKeyArgValues() throws MalformedURLException {
    init("one");

    MessageKey arg1 = RiskscapeMessage.of("arg1");
    assertThat(subject.getMessage("with.args", new Object[] {arg1}, JP),
        is("base bundle one argument one message"));
    assertThat(subject.getMessage("with.args", new Object[] {arg1}, NZ),
        is("base bundle one argument one message en_NZ"));

    MessageKey arg2 = RiskscapeMessage.ofDefault("default {0}", "d1");
    assertThat(subject.getMessage(RiskscapeMessage.ofDefault("outer {0}", arg2)),
        is("outer default d1"));

  }

  @Test
  public void oneBundleIsEnough() throws MalformedURLException {
    init("one");
    //Italy will get base bundle
    assertThat(subject.getMessage("name", null, JP), is("one"));
    assertThat(subject.getMessage("description", null, JP), is("base bundle one"));
    assertThat(subject.getMessage("with.args", new Object[] {"my-arg"}, JP), is("base bundle one my-arg"));

    assertThat(subject.getMessage("name", null, NZ), is("one en_NZ"));
    assertThat(subject.getMessage("description", null, NZ), is("base bundle one en_NZ"));
    assertThat(subject.getMessage("with.args", new Object[] {"my-arg"}, NZ), is("base bundle one my-arg en_NZ"));

    assertThat(subject.getMessage("name", null, NZ_V1), is("one en_NZ_variant1"));
    assertThat(subject.getMessage("description", null, NZ_V1), is("base bundle one en_NZ_variant1"));
    assertThat(subject.getMessage("with.args", new Object[] {"my-arg"}, NZ_V1),
        is("base bundle one my-arg en_NZ_variant1"));
  }

  @Test
  public void canBeBackedByASimplerResourceLoaderClasspath() throws Exception {
    subject = new RiskscapeMessageSource("messages", newResourceClassLoader("one"));

  // copy-pastad test assertions from the oneBundleIsEnough test above
    assertThat(subject.getMessage("name", null, JP), is("one"));
    assertThat(subject.getMessage("description", null, JP), is("base bundle one"));
    assertThat(subject.getMessage("with.args", new Object[] {"my-arg"}, JP), is("base bundle one my-arg"));

    assertThat(subject.getMessage("name", null, NZ), is("one en_NZ"));
    assertThat(subject.getMessage("description", null, NZ), is("base bundle one en_NZ"));
    assertThat(subject.getMessage("with.args", new Object[] {"my-arg"}, NZ), is("base bundle one my-arg en_NZ"));

    assertThat(subject.getMessage("name", null, NZ_V1), is("one en_NZ_variant1"));
    assertThat(subject.getMessage("description", null, NZ_V1), is("base bundle one en_NZ_variant1"));
    assertThat(subject.getMessage("with.args", new Object[] {"my-arg"}, NZ_V1),
        is("base bundle one my-arg en_NZ_variant1"));
  }

  @Test
  public void classpathOrderWins() throws MalformedURLException {
    init("two");
    init("one");
    assertThat(subject.getMessage("name", null, JP), is("two"));
    // there's no nz locale in two, but there is in one, which is more specific
    assertThat(subject.getMessage("name", null, NZ), is("one en_NZ"));
    // two has a key for the variant1
    assertThat(subject.getMessage("name", null, NZ_V1), is("two en_NZ_variant1"));

    //Fall back to bundle one, description is not in bundle two
    assertThat(subject.getMessage("description", null, JP), is("base bundle one"));
    assertThat(subject.getMessage("description", null, NZ), is("base bundle one en_NZ"));
    assertThat(subject.getMessage("description", null, NZ_V1), is("base bundle one en_NZ_variant1"));
  }

  @Test
  public void defaultOnlyIsOkay() throws MalformedURLException {
    init("one");

    assertThat(subject.getMessage(RiskscapeMessage.ofDefault("default message")), is("default message"));
  }

  @Test
  public void defaultMessageCanBeFormatWithArgs() throws MalformedURLException {
    init("one");

    assertThat(subject.getMessage(RiskscapeMessage.ofDefault("default message {1} {0}", "one", "two")),
        is("default message two one"));

    assertThat(
        subject.getMessage(RiskscapeMessage.withDefault("unknown-code", "default message {1} {0}", "one", "two")),
        is("default message two one"));
  }

}
