/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.i18n;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Enumeration;
import java.util.List;

import org.junit.Test;

import com.google.common.base.Charsets;
import com.google.common.collect.Lists;
import com.google.common.io.ByteSource;


public class ResourceClassLoaderTest {

  @Test
  public void canFindResult_FromSingleURLRoot() throws Exception {
    URL set1 = ResourceClassLoaderTest.class.getResource("/nz/org/riskscape/engine/i18n/ClassLoaderTests/set1/");
    ClassLoader subject = new ResourceClassLoader("", set1);
    assertThat(subject.getResource("file.txt"), is(notNullValue()));

    //And as a stream
    assertThat(read(subject.getResourceAsStream("file.txt")), is("set1"));
  }

  @Test
  public void canFindResult_FromManyURLRoots() throws Exception {
    URL set1 = ResourceClassLoaderTest.class.getResource("/nz/org/riskscape/engine/i18n/ClassLoaderTests/set1/");
    URL set2 = ResourceClassLoaderTest.class.getResource("/nz/org/riskscape/engine/i18n/ClassLoaderTests/set2/");
    ClassLoader subject = new ResourceClassLoader("", set2, set1);

    //Finds file.txt in set2, it is first in URL list
    assertThat(subject.getResource("file.txt"), is(notNullValue()));
    //And as a stream
    assertThat(read(subject.getResourceAsStream("file.txt")), is("set2"));

    //but still finds resource that is only in set1
    assertThat(subject.getResource("only-set1.txt"), is(notNullValue()));
    //And as a stream
    assertThat(read(subject.getResourceAsStream("only-set1.txt")), is("only-set1"));

    Enumeration<URL> all = subject.getResources("file.txt");
    List<String> found = Lists.newArrayList();
    while(all.hasMoreElements()) {
      found.add(read(all.nextElement().openStream()));
    }
    assertThat(found, contains(
        is("set2"),
        is("set1")
    ));
  }

  @Test
  public void noUrlsIsFine_ButNothingGetsFound() {
    ClassLoader subject = new ResourceClassLoader("");
    assertThat(subject.getResource(""), is(nullValue()));

    assertThat(subject.getResource("/nz/org/riskscape/engine/i18n/ClassLoaderTests"), is(nullValue()));
    //The following assert is to ensure the file being looked for, does exist in the parent class loader
    assertThat(ResourceClassLoaderTest.class.getResource("/nz/org/riskscape/engine/i18n/ClassLoaderTests"),
        is(notNullValue()));
  }

  String read(InputStream is) throws Exception {
    ByteSource byteSource = new ByteSource() {
      @Override
      public InputStream openStream() throws IOException {
        return is;
      }
    };
    return byteSource.asCharSource(Charsets.UTF_8).read().trim();
  }

}
