/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.i18n;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.net.URL;
import java.util.Locale;
import java.util.ResourceBundle;

import org.junit.Test;

public class MergedPropertiesResourceBundleTest {

  I18nClassLoader classLoader = new I18nClassLoader();

  ResourceBundleControl control = ResourceBundleControl.INSTANCE;

  public MergedPropertiesResourceBundleTest() {
    //Need to setup our custom class loader.
    classLoader = new I18nClassLoader();

    classLoader.append("set1", new ResourceClassLoader("set1/", new URL[] {
      MergedPropertiesResourceBundleTest.class
      .getResource("/nz/org/riskscape/engine/i18n/MergedPropertiesResourceBundleTest/")
    }));

    classLoader.insertBefore("set4", new ResourceClassLoader("", new URL[] {
      MergedPropertiesResourceBundleTest.class
      .getResource("/nz/org/riskscape/engine/i18n/MergedPropertiesResourceBundleTest/set4/")
    }), (o, c) -> "set1".equals(o));
    classLoader.insertBefore("set3", new ResourceClassLoader("", new URL[] {
      MergedPropertiesResourceBundleTest.class
      .getResource("/nz/org/riskscape/engine/i18n/MergedPropertiesResourceBundleTest/set3/")
    }), (o, c) -> "set1".equals(o));
    classLoader.insertBefore("set2", new ResourceClassLoader("set2/", new URL[] {
      MergedPropertiesResourceBundleTest.class
      .getResource("/nz/org/riskscape/engine/i18n/MergedPropertiesResourceBundleTest/")
    }), (o, c) -> "set1".equals(o));
  }

  @Test
  public void canGetBaseNameOnlyInSet1() {
    ResourceBundle bundle = ResourceBundle.getBundle("only-in-set1",Locale.FRENCH, classLoader,
        control);
    assertThat(bundle.getString("name"), is("only in set one"));
  }

  @Test
  public void canGetBaseNameOnlyInSet2() {
    ResourceBundle bundle = ResourceBundle.getBundle("only-in-set2",Locale.FRENCH, classLoader,
        control);
    assertThat(bundle.getString("name"), is("only in set two"));
  }

  @Test
  public void canGetBaseNameAll() {
    ResourceBundle bundle = ResourceBundle.getBundle("all",Locale.FRENCH, classLoader,
        control);
    //Last one wins
    assertThat(bundle.getString("name"), is("all, set3"));

    assertThat(bundle.getString("set1"), is("this is set1"));
    assertThat(bundle.getString("set2"), is("this is set2"));
    assertThat(bundle.getString("set3"), is("this is set3"));
  }

  @Test
  public void mergesSets_LastWins() {
    ResourceBundle bundle = ResourceBundle.getBundle("in-set1-set3",Locale.ENGLISH, classLoader,
        control);
    //Set3 is last in the class loader
    assertThat(bundle.getString("name"), is("this is set3"));

    assertThat(bundle.getString("only_set1"), is("set1"));
    assertThat(bundle.getString("only-set3"), is("set3"));

    bundle = ResourceBundle.getBundle("in-set1-set3",Locale.FRENCH, classLoader,
        control);
    //but french in set1 is more specific locale match
    assertThat(bundle.getString("name"), is("french set1"));
  }

  @Test
  public void getKeysEnumeratesAllKeys_WhenBundlesAllMerged() {
    ResourceBundle bundle = ResourceBundle.getBundle("in-set1-set3",Locale.ENGLISH, classLoader,
        control);

    //Testing that the returned bundle has keys for each distinct key used in all the properties
    //files(in different paths) that make up in-set1-set2
    assertThat(bundle.keySet(), containsInAnyOrder("name", "only_set1", "only-set3"));
  }

  @Test
  public void getKeysEnumeratesAllKeys_FromManyLocales() {
    ResourceBundle bundle = ResourceBundle.getBundle("set4",Locale.FRENCH, classLoader,
        control);

    //Testing that the returned bundle contains all the distinct keys that are in used in the diffenct
    //locale specific bundles being scanned. In this case French and the Root bundle.
    assertThat(bundle.keySet(), containsInAnyOrder("name", "description"));
  }

}
