/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.i18n;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.stream.Collectors;

import org.junit.Before;
import org.junit.Test;

import com.google.common.base.Charsets;
import com.google.common.io.ByteSource;

public class I18nClassLoaderTest {

  I18nClassLoader subject;

  @Before
  public void setup() {
    subject = new I18nClassLoader();
  }

  ResourceClassLoader set1And2 = new ResourceClassLoader("",
      I18nClassLoaderTest.class.getResource("/nz/org/riskscape/engine/i18n/ClassLoaderTests/set1/"),
      I18nClassLoaderTest.class.getResource("/nz/org/riskscape/engine/i18n/ClassLoaderTests/set2/")
  );

  //Will return file from set2 then set1
  ResourceClassLoader set2And1 = new ResourceClassLoader("",
      I18nClassLoaderTest.class.getResource("/nz/org/riskscape/engine/i18n/ClassLoaderTests/set2/"),
      I18nClassLoaderTest.class.getResource("/nz/org/riskscape/engine/i18n/ClassLoaderTests/set1/")
  );

  ResourceClassLoader set1 = new ResourceClassLoader("",
      I18nClassLoaderTest.class.getResource("/nz/org/riskscape/engine/i18n/ClassLoaderTests/set1/"));

  ResourceClassLoader set2 = new ResourceClassLoader("",
      I18nClassLoaderTest.class.getResource("/nz/org/riskscape/engine/i18n/ClassLoaderTests/set2/"));



  @Test
  public void testGetResourcesByPrecedence() throws Exception {
    //Will return file from set1 then set2

    subject.append("set1", set1And2);
    assertThat(subject.getResourcesByPrecedence("file.txt").stream()
        .map(u -> read(u))
        .collect(Collectors.toList()),
        contains("set1")
    );

    subject.append("set2", set2And1);
    assertThat(subject.getResourcesByPrecedence("file.txt").stream()
        .map(u -> read(u))
        .collect(Collectors.toList()),
        contains("set1", "set2")
    );

    //You can add the same class loader many times if you want to. Convenient for this test
    subject.unshift("set2", set2And1);
    assertThat(subject.getResourcesByPrecedence("file.txt").stream()
        .map(u -> read(u))
        .collect(Collectors.toList()),
        contains("set2", "set1", "set2")
    );
  }

  @Test
  public void testUnshift() throws Exception {

    //prime the class loader with set1
    subject.append("set1", set1);

    //then unshift set2 to take precedence
    subject.unshift("set2", set2);
    //Can read from set2 now, and set2 has precedence
    assertThat(read(subject.getResourceAsStream("only-set2.txt")), is("only-set2"));
    assertThat(read(subject.getResourceAsStream("file.txt")), is("set2"));

    //can still get resource from set1
    assertThat(read(subject.getResourceAsStream("only-set1.txt")), is("only-set1"));
  }

  @Test
  public void testUnshift_FromEmpty() throws Exception {

    //unshift will work fine on from empty
    subject.unshift("set2", set2);
    //Can read from set2 now, and set2 has precedence
    assertThat(read(subject.getResourceAsStream("only-set2.txt")), is("only-set2"));
    assertThat(read(subject.getResourceAsStream("file.txt")), is("set2"));

    //set1 is not available
    assertThat(subject.getResourceAsStream("only-set1.txt"), is(nullValue()));
  }

  @Test
  public void testInsertBefore() throws Exception {

    //insert before/after methods return false if not matched, and nothing is inserted
    //these two will aways fail because there are no existing entries yet
    assertThat(subject.insertBefore("set1", set1, (o, c) -> true), is(false));
    //check that nothing gets found
    assertThat(subject.getResource("file.txt"), is(nullValue()));

    subject.append("set1", set1);
    assertThat(subject.getResource("file.txt"), is(notNullValue()));

    //adding set2 will fail if the test is bad.
    assertThat(subject.insertBefore("set2", set2, (o, c) -> false), is(false));
    assertThat(read(subject.getResourceAsStream("file.txt")), is("set1"));

    assertThat(subject.insertBefore("set2", set2, (o, c) -> c == set1), is(true));
    //Can read from set2 now, and set2 has precedence
    assertThat(read(subject.getResourceAsStream("only-set2.txt")), is("only-set2"));
    assertThat(read(subject.getResourceAsStream("file.txt")), is("set2"));

    //can still get resource from set1
    assertThat(read(subject.getResourceAsStream("only-set1.txt")), is("only-set1"));
  }

  @Test
  public void testInsertAfter() throws Exception {

    //insert before/after methods return false if not matched, and nothing is inserted
    //these two will aways fail because there are no existing entries yet
    assertThat(subject.insertAfter("set1", set1, (o, c) -> true), is(false));
    //check that nothing gets found
    assertThat(subject.getResource("file.txt"), is(nullValue()));

    subject.append("set1", set1);
    assertThat(subject.getResource("file.txt"), is(notNullValue()));

    //adding set2 will fail if the test is bad.
    assertThat(subject.insertBefore("set2", set2, (o, c) -> false), is(false));
    assertThat(read(subject.getResourceAsStream("file.txt")), is("set1"));

    assertThat(subject.insertAfter("set2", set2, (o, c) -> c == set1), is(true));
    //Can read from set2 now
    assertThat(read(subject.getResourceAsStream("only-set2.txt")), is("only-set2"));
    //but set1 has precedence
    assertThat(read(subject.getResourceAsStream("file.txt")), is("set1"));

  }

  String read(URL url) {
    try {
    return read(url.openStream());
    } catch (Exception e) {
      return null;
    }
  }

  String read(InputStream is) throws Exception {
    ByteSource byteSource = new ByteSource() {
      @Override
      public InputStream openStream() throws IOException {
        return is;
      }
    };
    return byteSource.asCharSource(Charsets.UTF_8).read().trim();
  }
}
